import React, { useState, useRef, useEffect } from 'react';
import {
  Box,
  VStack,
  HStack,
  Text,
  Button,
  ButtonText,
  Input,
  InputField,
  ScrollView,
  Image,
  Pressable,
  Card,
  Spinner,
} from '@gluestack-ui/themed';
import { StatusBar, Alert, KeyboardAvoidingView, Platform, TouchableOpacity } from 'react-native';
import { SafeAreaView } from 'react-native-safe-area-context';
import { useNavigation } from '@react-navigation/native';
import { useDispatch, useSelector } from 'react-redux';
import { useTheme } from '../contexts/ThemeContext';
import { addMessage, sendChatMessage, clearError } from '../store/slices/chatSlice';
import { RootState } from '../store';
import { NewsArticle } from '../types';

const ChatScreen: React.FC = () => {
  const navigation = useNavigation();
  const dispatch = useDispatch();
  const { isDark, colors } = useTheme();
  const { messages, isLoading, error } = useSelector((state: RootState) => state.chat);
  
  const [inputText, setInputText] = useState('');
  const scrollViewRef = useRef<ScrollView>(null);

  const categories = [
    { key: 'Politics', name: 'Politics' },
    { key: 'Business', name: 'Business' },
    { key: 'Sports', name: 'Sports' },
    { key: 'Entertainment', name: 'Entertainment' },
    { key: 'Technology', name: 'Technology' },
    { key: 'Health', name: 'Health' },
  ];

  useEffect(() => {
    if (error) {
      Alert.alert('Error', error);
      dispatch(clearError());
    }
  }, [error, dispatch]);

  const handleSendMessage = (message?: string) => {
    const textToSend = message || inputText.trim();
    if (!textToSend) return;

    // Add user message immediately
    const userMessage = {
      id: Date.now().toString(),
      text: textToSend,
      isUser: true,
      timestamp: Date.now(),
    };
    dispatch(addMessage(userMessage));

    // Send to API
    dispatch(sendChatMessage(textToSend));

    // Clear input
    setInputText('');

    // Scroll to bottom
    setTimeout(() => {
      scrollViewRef.current?.scrollToEnd({ animated: true });
    }, 100);
  };

  const handleNewsPress = (article: NewsArticle) => {
    navigation.navigate('DeepDive' as never, { articleId: article.id.toString() } as never);
  };

  const handleCategoryPress = async (category: string) => {
    const categoryQueries = {
      'Politics': 'What are the latest political news and developments?',
      'Business': 'What are the latest business and economic news?',
      'Sports': 'What are the latest sports news?',
      'Entertainment': 'What are the latest entertainment and celebrity news?',
      'Technology': 'What are the latest technology news?',
      'Health': 'What are the latest health and medical news?'
    };
    
    const query = categoryQueries[category as keyof typeof categoryQueries];
    if (query) {
      handleSendMessage(query);
    }
  };

  const renderNewsCard = (article: NewsArticle) => (
    <Card key={article.id} p="$3" m="$2" bg={colors.surface} borderRadius="$lg" borderWidth={1} borderColor={colors.border}>
      <Pressable onPress={() => handleNewsPress(article)}>
        <VStack space="sm">
          <Image
            source={{ uri: article.post_image || 'https://via.placeholder.com/400x200?text=No+Image' }}
            alt={article.post_title}
            width="100%"
            height={120}
            borderRadius="$md"
          />
          
          <VStack space="xs">
            <Text fontSize="$sm" fontWeight="$bold" color={colors.text} numberOfLines={2}>
              {article.post_title}
            </Text>
            
            <Text fontSize="$xs" color={colors.textSecondary} numberOfLines={2}>
              {article.post_excerpt}
            </Text>
            
            <HStack justifyContent="space-between" alignItems="center">
              <Text fontSize="$xs" color={colors.textSecondary}>
                {new Date(article.date_added).toLocaleDateString()}
              </Text>
              <Text fontSize="$xs" color={colors.primary}>
                Read more →
              </Text>
            </HStack>
          </VStack>
        </VStack>
      </Pressable>
    </Card>
  );

  const renderMessage = (message: any) => (
    <Box key={message.id} p="$3">
      <HStack space="sm" justifyContent={message.isUser ? 'flex-end' : 'flex-start'}>
        {!message.isUser && (
          <Box
            width={32}
            height={32}
            borderRadius="$full"
            bg={colors.primary}
            justifyContent="center"
            alignItems="center"
          >
            <Text color="#ffffff" fontSize="$sm" fontWeight="$bold">
              AI
            </Text>
          </Box>
        )}
        
        <VStack 
          maxWidth="80%" 
          space="sm"
          alignItems={message.isUser ? 'flex-end' : 'flex-start'}
        >
          <Box
            bg={message.isUser ? colors.primary : colors.surface}
            borderRadius="$lg"
            p="$3"
            borderWidth={1}
            borderColor={message.isUser ? colors.primary : colors.border}
          >
            <Text 
              color={message.isUser ? '#ffffff' : colors.text}
              fontSize="$sm"
              lineHeight="$md"
            >
              {message.text}
            </Text>
          </Box>
          
          {message.news_data && message.news_data.length > 0 && (
            <VStack space="sm" width="100%">
              <Text fontSize="$xs" color={colors.textSecondary} fontWeight="$medium">
                Related News:
              </Text>
              {message.news_data.map(renderNewsCard)}
            </VStack>
          )}
        </VStack>
        
        {message.isUser && (
          <Box
            width={32}
            height={32}
            borderRadius="$full"
            bg={colors.textSecondary}
            justifyContent="center"
            alignItems="center"
          >
            <Text color={colors.background} fontSize="$sm" fontWeight="$bold">
              U
            </Text>
          </Box>
        )}
      </HStack>
    </Box>
  );

  return (
    <SafeAreaView style={{ flex: 1, backgroundColor: colors.background }}>
      <StatusBar barStyle={isDark ? 'light-content' : 'dark-content'} />
      
      <Box flex={1} bg={colors.background}>
        {/* Header */}
        <HStack 
          justifyContent="space-between" 
          alignItems="center" 
          p="$4"
          borderBottomWidth={1}
          borderBottomColor={colors.border}
          bg={colors.background}
        >
          <Pressable onPress={() => navigation.goBack()}>
            <Box 
              width={40} 
              height={40} 
              borderRadius="$full" 
              bg={colors.surface}
              justifyContent="center"
              alignItems="center"
              borderWidth={1}
              borderColor={colors.border}
            >
              <Text fontSize="$lg" color={colors.text} fontWeight="$bold">
                ←
              </Text>
            </Box>
          </Pressable>
          
          <Text fontSize="$lg" fontWeight="$bold" color={colors.text}>
            Mews AI
          </Text>
          
          <Pressable onPress={() => {}}>
            <Box 
              width={40} 
              height={40} 
              borderRadius="$full" 
              bg={colors.surface}
              justifyContent="center"
              alignItems="center"
              borderWidth={1}
              borderColor={colors.border}
            >
              <Text fontSize="$lg" color={colors.text} fontWeight="$bold">
                N
              </Text>
            </Box>
          </Pressable>
        </HStack>

        {/* Category Buttons */}
        <Box p="$4" borderBottomWidth={1} borderBottomColor={colors.border}>
          <ScrollView horizontal showsHorizontalScrollIndicator={false}>
            <HStack space="sm">
              {categories.map((category) => (
                <Button
                  key={category.key}
                  variant="outline"
                  size="sm"
                  onPress={() => handleCategoryPress(category.name)}
                  bg={colors.surface}
                  borderColor={colors.border}
                  borderRadius="$full"
                  px="$4"
                  py="$2"
                >
                  <ButtonText color={colors.text} fontSize="$sm" fontWeight="$medium">
                    {category.name}
                  </ButtonText>
                </Button>
              ))}
            </HStack>
          </ScrollView>
        </Box>

        {/* Messages */}
        <KeyboardAvoidingView 
          style={{ flex: 1 }} 
          behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        >
          <ScrollView 
            ref={scrollViewRef}
            flex={1}
            showsVerticalScrollIndicator={false}
            onContentSizeChange={() => scrollViewRef.current?.scrollToEnd({ animated: true })}
          >
            {messages.map(renderMessage)}
            
            {isLoading && (
              <Box p="$3">
                <HStack space="sm" alignItems="center">
                  <Box
                    width={32}
                    height={32}
                    borderRadius="$full"
                    bg={colors.primary}
                    justifyContent="center"
                    alignItems="center"
                  >
                    <Text color="#ffffff" fontSize="$sm" fontWeight="$bold">
                      AI
                    </Text>
                  </Box>
                  <Box
                    bg={colors.surface}
                    borderRadius="$lg"
                    p="$3"
                    borderWidth={1}
                    borderColor={colors.border}
                  >
                    <HStack space="sm" alignItems="center">
                      <Spinner size="small" color={colors.primary} />
                      <Text color={colors.text} fontSize="$sm">
                        Thinking...
                      </Text>
                    </HStack>
                  </Box>
                </HStack>
              </Box>
            )}
          </ScrollView>

          {/* Input Area */}
          <Box p="$4" borderTopWidth={1} borderTopColor={colors.border}>
            <HStack space="sm" alignItems="center">
              <Input flex={1} variant="outline" size="lg">
                <InputField
                  placeholder="Ask me anything..."
                  placeholderTextColor={colors.textSecondary}
                  value={inputText}
                  onChangeText={setInputText}
                  color={colors.text}
                  onSubmitEditing={() => handleSendMessage()}
                  returnKeyType="send"
                />
              </Input>
              
              <Button
                onPress={() => handleSendMessage()}
                disabled={!inputText.trim() || isLoading}
                bg={colors.primary}
                borderRadius="$full"
                width={48}
                height={48}
              >
                <Text color="#ffffff" fontSize="$lg" fontWeight="$bold">
                  →
                </Text>
              </Button>
            </HStack>
          </Box>
        </KeyboardAvoidingView>
      </Box>
    </SafeAreaView>
  );
};

export default ChatScreen;