import axios from 'axios';

const BASE_URL = 'http://192.168.100.227/mews/public';

export interface NewsArticle {
  id: number;
  post_title: string;
  post_excerpt: string;
  post_content: string;
  post_image: string;
  post_category: string;
  post_source: string;
  date_added: string;
  views: number;
}

export interface ChatMessage {
  id: string;
  text: string;
  isUser: boolean;
  timestamp: Date;
  news_data?: NewsArticle[];
}

export interface ChatResponse {
  success: boolean;
  message: string;
  news_data?: NewsArticle[];
}

class ApiService {
  private baseURL: string;

  constructor() {
    this.baseURL = BASE_URL;
  }

  async sendChatMessage(message: string): Promise<ChatResponse> {
    try {
      const response = await axios.post(`${this.baseURL}/api/chat`, {
        message,
      });
      return response.data;
    } catch (error) {
      console.error('Error sending chat message:', error);
      throw error;
    }
  }

  async getLatestNews(): Promise<NewsArticle[]> {
    try {
      const response = await axios.get(`${this.baseURL}/api/news`);
      return response.data.news || [];
    } catch (error) {
      console.error('Error fetching latest news:', error);
      return [];
    }
  }

  async getNewsByCategory(category: string): Promise<NewsArticle[]> {
    try {
      const response = await axios.get(`${this.baseURL}/api/news/categories/${category}`);
      return response.data.news || [];
    } catch (error) {
      console.error('Error fetching news by category:', error);
      return [];
    }
  }

  async getNewsCategories(): Promise<string[]> {
    try {
      const response = await axios.get(`${this.baseURL}/api/news/categories`);
      return response.data.categories || [];
    } catch (error) {
      console.error('Error fetching news categories:', error);
      return [];
    }
  }

  async getArticle(articleId: string): Promise<NewsArticle | null> {
    try {
      const response = await axios.get(`${this.baseURL}/api/news/deep-dive/${articleId}`);
      return response.data.article || null;
    } catch (error) {
      console.error('Error fetching article:', error);
      return null;
    }
  }

  async askQuestion(articleId: string, question: string): Promise<string> {
    try {
      const response = await axios.post(`${this.baseURL}/api/news/deep-dive/${articleId}/chat`, {
        question,
      });
      return response.data.answer || '';
    } catch (error) {
      console.error('Error asking question:', error);
      return 'Sorry, I could not process your question.';
    }
  }

  async searchNews(query: string): Promise<NewsArticle[]> {
    try {
      const response = await axios.get(`${this.baseURL}/api/news/search`, {
        params: { q: query },
      });
      return response.data.news || [];
    } catch (error) {
      console.error('Error searching news:', error);
      return [];
    }
  }
}

export const apiService = new ApiService();