import { createSlice, createAsyncThunk, PayloadAction } from '@reduxjs/toolkit';
import { ChatMessage, NewsArticle } from '../../types';
import { apiService } from '../../services/api';

interface ChatState {
  messages: ChatMessage[];
  isLoading: boolean;
  error: string | null;
}

const initialState: ChatState = {
  messages: [
    {
      id: 'welcome',
      text: 'Hello! I\'m Mews AI, your intelligent news assistant. Ask me about the latest news, or explore different categories.',
      isUser: false,
      timestamp: Date.now(),
    },
  ],
  isLoading: false,
  error: null,
};

export const sendChatMessage = createAsyncThunk(
  'chat/sendMessage',
  async (message: string, { rejectWithValue }) => {
    try {
      const response = await apiService.sendChatMessage(message);
      return response;
    } catch (error) {
      return rejectWithValue('Failed to send message. Please try again.');
    }
  }
);

const chatSlice = createSlice({
  name: 'chat',
  initialState,
  reducers: {
    addMessage: (state, action: PayloadAction<ChatMessage>) => {
      state.messages.push(action.payload);
    },
    clearMessages: (state) => {
      state.messages = [state.messages[0]]; // Keep welcome message
    },
    clearError: (state) => {
      state.error = null;
    },
  },
  extraReducers: (builder) => {
    builder
      .addCase(sendChatMessage.pending, (state) => {
        state.isLoading = true;
        state.error = null;
      })
      .addCase(sendChatMessage.fulfilled, (state, action) => {
        state.isLoading = false;
        const aiMessage: ChatMessage = {
          id: Date.now().toString(),
          text: action.payload.message,
          isUser: false,
          timestamp: Date.now(),
          news_data: action.payload.news_data,
        };
        state.messages.push(aiMessage);
      })
      .addCase(sendChatMessage.rejected, (state, action) => {
        state.isLoading = false;
        state.error = action.payload as string;
      });
  },
});

export const { addMessage, clearMessages, clearError } = chatSlice.actions;
export default chatSlice.reducer;