import React, { useState, useRef, useEffect } from 'react';
import {
  View,
  StyleSheet,
  ScrollView,
  TouchableOpacity,
  KeyboardAvoidingView,
  Platform,
  Alert,
  Image,
  StatusBar,
  TouchableWithoutFeedback,
  Keyboard,
} from 'react-native';
import { useDispatch, useSelector } from 'react-redux';
import {
  TextInput,
  Text,
  ActivityIndicator,
} from 'react-native-paper';
import { useRoute, RouteProp, useNavigation } from '@react-navigation/native';
import { RootState, AppDispatch } from '../store';
import { sendChatMessage } from '../store/slices/chatSlice';
import { NewsArticle } from '../types';

type RootStackParamList = {
  Article: { article: NewsArticle };
};

type ArticleScreenRouteProp = RouteProp<RootStackParamList, 'Article'>;

const ArticleScreen: React.FC = () => {
  const route = useRoute<ArticleScreenRouteProp>();
  const navigation = useNavigation();
  const { article } = route.params;
  const dispatch = useDispatch<AppDispatch>();
  const { messages, isLoading } = useSelector((state: RootState) => state.chat);
  const [inputText, setInputText] = useState('');
  const scrollViewRef = useRef<ScrollView>(null);

  useEffect(() => {
    // Auto-scroll to bottom when new messages arrive
    if (messages.length > 0) {
      setTimeout(() => {
        scrollViewRef.current?.scrollToEnd({ animated: true });
      }, 100);
    }
  }, [messages]);

  const handleSendMessage = async () => {
    if (inputText.trim() === '' || isLoading) return;

    const message = inputText.trim();
    setInputText('');

    try {
      await dispatch(sendChatMessage(message));
    } catch (error) {
      Alert.alert('Error', 'Failed to send message. Please try again.');
    }
  };

  const renderMessage = ({ item, index }: { item: any; index: number }) => (
    <View style={[
      styles.messageContainer,
      item.isUser ? styles.userMessageContainer : styles.aiMessageContainer
    ]}>
      <View style={[
        styles.messageBubble,
        item.isUser ? styles.userMessageBubble : styles.aiMessageBubble
      ]}>
        <Text style={[
          styles.messageText,
          item.isUser ? styles.userMessageText : styles.aiMessageText
        ]}>
          {item.text}
        </Text>
        
        {!item.isUser && item.intent && (
          <View style={styles.intentBadge}>
            <Text style={styles.intentText}>{item.intent}</Text>
          </View>
        )}
        
        {!item.isUser && item.newsData && item.newsData.length > 0 && (
          <View style={styles.newsDataContainer}>
            <Text style={styles.newsDataTitle}>Related News:</Text>
            {item.newsData.map((news: any, index: number) => (
              <View key={`related-news-${news.id || index}`} style={styles.newsItem}>
                <Text style={styles.newsTitle}>{news.title}</Text>
                <Text style={styles.newsSource}>{news.source}</Text>
              </View>
            ))}
          </View>
        )}
      </View>
    </View>
  );

  const renderLoadingIndicator = () => (
    <View style={styles.loadingContainer}>
      <ActivityIndicator size="small" color="#0a57d4" />
      <Text style={styles.loadingText}>AI is thinking...</Text>
    </View>
  );

  return (
    <TouchableWithoutFeedback onPress={Keyboard.dismiss}>
      <View style={styles.container}>
        <StatusBar barStyle="dark-content" backgroundColor="#ffffff" />
      
      {/* Header with back button */}
      <View style={styles.header}>
        <TouchableOpacity 
          style={styles.backButton}
          onPress={() => navigation.goBack()}
          activeOpacity={0.7}
        >
          <Text style={styles.backButtonText}>← Back</Text>
        </TouchableOpacity>
        <Text style={styles.headerTitle}>Deep Dive</Text>
        <View style={styles.headerSpacer} />
      </View>

      <ScrollView style={styles.scrollContainer} showsVerticalScrollIndicator={false}>
        {/* Article Image */}
        {article.post_image && article.post_image !== 'N/A' && article.post_image.trim() !== '' && (
          <View style={styles.articleImageContainer}>
            <Image
              source={{ 
                uri: article.post_image.startsWith('http') 
                  ? article.post_image 
                  : `http://172.20.10.4/mews/public${article.post_image}`
              }}
              style={styles.articleImage}
              resizeMode="cover"
              onError={(error) => console.log('Image load error:', error)}
            />
            <View style={styles.articleCategoryOverlay}>
              <Text style={styles.articleCategoryText}>{article.post_category}</Text>
            </View>
          </View>
        )}

        {/* Article Content */}
        <View style={styles.articleContent}>
          <Text style={styles.articleTitle}>{article.post_title}</Text>
          
          <View style={styles.articleMeta}>
            <View style={styles.articleMetaItem}>
              <Text style={styles.articleMetaIcon}>📅</Text>
              <Text style={styles.articleMetaText}>
                {new Date(article.date_added).toLocaleDateString()}
              </Text>
            </View>
            <View style={styles.articleMetaItem}>
              <Text style={styles.articleMetaIcon}>👁️</Text>
              <Text style={styles.articleMetaText}>{article.views} views</Text>
            </View>
            <View style={styles.articleMetaItem}>
              <Text style={styles.articleMetaIcon}>📰</Text>
              <Text style={styles.articleMetaText}>{article.post_source}</Text>
            </View>
          </View>

          <Text style={styles.articleText}>
            {article.post_content ? 
              article.post_content.replace(/<[^>]*>/g, '') : 
              'No content available'
            }
          </Text>
        </View>

        {/* Chat Section */}
        <View style={styles.chatSection}>
          <View style={styles.chatHeader}>
            <Text style={styles.chatTitle}>💬 Ask about this article</Text>
            <Text style={styles.chatSubtitle}>Get AI insights and analysis</Text>
          </View>
          
          <View style={styles.chatMessages}>
            {messages.length === 0 ? (
              <View style={styles.emptyChatContainer}>
                <Text style={styles.emptyChatText}>
                  Ask me anything about this article! I can provide analysis, context, or answer specific questions.
                </Text>
              </View>
            ) : (
              messages.map((message, index) => (
                <View key={message.id || `message-${index}`}>
                  {renderMessage({ item: message, index })}
                </View>
              ))
            )}
            {isLoading && renderLoadingIndicator()}
          </View>

          {/* Input Area */}
          <KeyboardAvoidingView
            behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
            style={styles.inputContainer}
          >
            <View style={styles.inputRow}>
              <TextInput
                style={styles.textInput}
                value={inputText}
                onChangeText={setInputText}
                placeholder="Ask about this article..."
                placeholderTextColor="#999"
                multiline
                maxLength={500}
                disabled={isLoading}
                mode="outlined"
                outlineColor="transparent"
                activeOutlineColor="transparent"
              />
              <TouchableOpacity
                style={[
                  styles.sendButton,
                  (isLoading || inputText.trim() === '') && styles.sendButtonDisabled
                ]}
                onPress={handleSendMessage}
                disabled={isLoading || inputText.trim() === ''}
                activeOpacity={0.7}
              >
                <Text style={styles.sendIcon}>✈️</Text>
              </TouchableOpacity>
            </View>
          </KeyboardAvoidingView>
        </View>
      </ScrollView>
      </View>
    </TouchableWithoutFeedback>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#ffffff',
  },
  header: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingTop: 50,
    paddingHorizontal: 20,
    paddingBottom: 15,
    backgroundColor: '#ffffff',
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
  },
  backButton: {
    paddingVertical: 8,
    paddingHorizontal: 12,
  },
  backButtonText: {
    fontSize: 16,
    color: '#0a57d4',
    fontWeight: '600',
  },
  headerTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#333',
  },
  headerSpacer: {
    width: 60,
  },
  scrollContainer: {
    flex: 1,
  },
  articleImageContainer: {
    position: 'relative',
    height: 250,
    overflow: 'hidden',
  },
  articleImage: {
    width: '100%',
    height: '100%',
  },
  articleCategoryOverlay: {
    position: 'absolute',
    top: 15,
    left: 15,
    backgroundColor: 'rgba(10, 87, 212, 0.9)',
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 12,
  },
  articleCategoryText: {
    color: '#ffffff',
    fontSize: 12,
    fontWeight: '600',
  },
  articleContent: {
    padding: 20,
  },
  articleTitle: {
    fontSize: 24,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 16,
    lineHeight: 32,
  },
  articleMeta: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    gap: 12,
    marginBottom: 20,
  },
  articleMetaItem: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: 6,
  },
  articleMetaIcon: {
    fontSize: 14,
  },
  articleMetaText: {
    fontSize: 14,
    color: '#666',
  },
  articleText: {
    fontSize: 16,
    lineHeight: 26,
    color: '#333',
  },
  chatSection: {
    backgroundColor: 'rgba(10, 87, 212, 0.05)',
    marginTop: 20,
    borderTopLeftRadius: 20,
    borderTopRightRadius: 20,
  },
  chatHeader: {
    padding: 20,
    borderBottomWidth: 1,
    borderBottomColor: 'rgba(10, 87, 212, 0.1)',
  },
  chatTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 4,
  },
  chatSubtitle: {
    fontSize: 14,
    color: '#666',
  },
  chatMessages: {
    minHeight: 200,
    maxHeight: 300,
    padding: 20,
  },
  emptyChatContainer: {
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 40,
  },
  emptyChatText: {
    fontSize: 16,
    color: '#666',
    textAlign: 'center',
    lineHeight: 24,
  },
  messageContainer: {
    marginBottom: 15,
  },
  userMessageContainer: {
    alignItems: 'flex-end',
  },
  aiMessageContainer: {
    alignItems: 'flex-start',
  },
  messageBubble: {
    maxWidth: '85%',
    padding: 15,
    borderRadius: 16,
    backgroundColor: 'rgba(255, 255, 255, 0.9)',
    borderWidth: 1,
    borderColor: 'rgba(10, 87, 212, 0.1)',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 2,
  },
  userMessageBubble: {
    backgroundColor: '#0a57d4',
    borderColor: '#0a57d4',
  },
  aiMessageBubble: {
    backgroundColor: 'rgba(255, 255, 255, 0.95)',
    borderColor: 'rgba(10, 87, 212, 0.1)',
  },
  messageText: {
    fontSize: 15,
    lineHeight: 22,
  },
  userMessageText: {
    color: '#fff',
  },
  aiMessageText: {
    color: '#333',
  },
  intentBadge: {
    marginTop: 8,
    alignSelf: 'flex-start',
    backgroundColor: 'rgba(10, 87, 212, 0.1)',
    paddingHorizontal: 8,
    paddingVertical: 4,
    borderRadius: 12,
  },
  intentText: {
    fontSize: 11,
    color: '#0a57d4',
    fontWeight: '600',
  },
  loadingContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    padding: 20,
  },
  loadingText: {
    marginLeft: 8,
    color: '#666',
    fontSize: 14,
  },
  inputContainer: {
    backgroundColor: '#ffffff',
    padding: 20,
    borderTopWidth: 1,
    borderTopColor: 'rgba(10, 87, 212, 0.1)',
  },
  inputRow: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: 'rgba(10, 87, 212, 0.05)',
    borderRadius: 25,
    paddingHorizontal: 20,
    paddingVertical: 12,
    borderWidth: 1,
    borderColor: 'rgba(10, 87, 212, 0.1)',
  },
  textInput: {
    flex: 1,
    marginRight: 12,
    backgroundColor: 'transparent',
    maxHeight: 100,
    color: '#333',
    fontSize: 16,
  },
  sendButton: {
    backgroundColor: '#0a57d4',
    width: 40,
    height: 40,
    borderRadius: 20,
    alignItems: 'center',
    justifyContent: 'center',
  },
  sendButtonDisabled: {
    backgroundColor: '#ccc',
  },
  sendIcon: {
    fontSize: 16,
    color: '#ffffff',
  },
});

export default ArticleScreen;