import React, { useState, useEffect } from 'react';
import {
  View,
  StyleSheet,
  ScrollView,
  TouchableOpacity,
  Text,
  SafeAreaView,
  StatusBar,
  Image,
  FlatList,
  RefreshControl,
  Dimensions,
  Alert,
} from 'react-native';
import { useNavigation } from '@react-navigation/native';
import { useTheme } from '../contexts/ThemeContext';
import { NewsArticle } from '../types';
import { apiService } from '../services/api';

const { width: screenWidth } = Dimensions.get('window');

interface Category {
  id: string;
  name: string;
  isActive: boolean;
}

const DiscoverScreen: React.FC = () => {
  const navigation = useNavigation();
  const { isDark } = useTheme();
  const [selectedCategory, setSelectedCategory] = useState('For You');
  const [newsData, setNewsData] = useState<NewsArticle[]>([]);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [isLoading, setIsLoading] = useState(true);
  const [availableCategories, setAvailableCategories] = useState<string[]>([]);
  const [searchQuery, setSearchQuery] = useState('');
  const [isSearching, setIsSearching] = useState(false);
  const [page, setPage] = useState(1);
  const [hasMoreData, setHasMoreData] = useState(true);
  const [isLoadingMore, setIsLoadingMore] = useState(false);

  // Generate categories dynamically from available categories
  const generateCategories = (): Category[] => {
    const baseCategories = [
      { id: 'for-you', name: 'For You' },
      { id: 'top-stories', name: 'Top Stories' },
    ];
    
    const dynamicCategories = availableCategories.map(cat => ({
      id: cat.toLowerCase().replace(/\s+/g, '-'),
      name: cat,
    }));
    
    return [...baseCategories, ...dynamicCategories].map(cat => ({
      ...cat,
      isActive: selectedCategory === cat.name,
    }));
  };

  const categories = generateCategories();

  useEffect(() => {
    fetchCategories();
    fetchNewsData(true); // Initial load
  }, []);

  useEffect(() => {
    setPage(1);
    setHasMoreData(true);
    fetchNewsData(true); // Reset and fetch new data
  }, [selectedCategory]);

  const fetchCategories = async () => {
    try {
      const categories = await apiService.getNewsCategories();
      setAvailableCategories(categories);
    } catch (error) {
      console.error('Failed to fetch categories:', error);
      // Fallback categories
      setAvailableCategories(['Politics', 'Business', 'Entertainment', 'Sports', 'Technology', 'Health', 'International']);
    }
  };

  const fetchNewsData = async (reset = false) => {
    try {
      if (reset) {
        setIsLoading(true);
        setPage(1);
        setHasMoreData(true);
      } else {
        setIsLoadingMore(true);
      }

      const currentPage = reset ? 1 : page;
      const limit = 10; // Load 10 items at a time for better performance

      let data: NewsArticle[] = [];

      try {
        if (searchQuery.trim()) {
          // If there's a search query, search for news
          data = await apiService.searchNews(searchQuery, limit);
        } else if (selectedCategory === 'For You') {
          // Fetch latest news for "For You" feed
          data = await apiService.getLatestNews(limit);
        } else if (selectedCategory === 'Top Stories') {
          // Fetch latest news as top stories
          data = await apiService.getLatestNews(limit);
        } else {
          // Fetch news by specific category
          data = await apiService.getNewsByCategory(selectedCategory, limit);
        }
      } catch (apiError) {
        console.error('API Error:', apiError);
        // Use fallback data if API fails
        data = getFallbackNewsData();
      }

      // Transform the data to match the expected format
      const transformedData = data.map(article => ({
        ...article,
        // Ensure all required fields are present
        post_title: article.post_title || article.title || 'Untitled Article',
        post_excerpt: article.post_excerpt || article.content?.substring(0, 150) + '...' || 'No excerpt available',
        post_content: article.post_content || article.content || '',
        post_image: article.post_image || article.image || 'https://via.placeholder.com/400x200?text=No+Image',
        post_category: article.post_category || article.category || 'General',
        post_source: article.post_source || article.source || 'Unknown Source',
        date_added: article.date_added || article.date || new Date().toISOString(),
        views: article.views || Math.floor(Math.random() * 1000) + 100,
      }));

      if (reset) {
        setNewsData(transformedData);
      } else {
        setNewsData(prevData => [...prevData, ...transformedData]);
      }

      // Check if we have more data to load
      setHasMoreData(transformedData.length === limit);
      
      if (!reset) {
        setPage(prevPage => prevPage + 1);
      }

    } catch (error) {
      console.error('Failed to fetch news:', error);
      // Use fallback data instead of showing error
      const fallbackData = getFallbackNewsData();
      if (reset) {
        setNewsData(fallbackData);
      } else {
        setNewsData(prevData => [...prevData, ...fallbackData]);
      }
      setHasMoreData(false);
    } finally {
      setIsLoading(false);
      setIsLoadingMore(false);
    }
  };

  const getFallbackNewsData = (): NewsArticle[] => {
    return [
      {
        id: 'fallback-1',
        post_title: 'Ghana\'s Economic Growth Shows Positive Signs',
        post_excerpt: 'Latest economic indicators suggest Ghana is on a path to recovery with improved GDP growth and stable inflation rates.',
        post_content: 'Full article content here...',
        post_image: 'https://via.placeholder.com/400x200/0a57d4/ffffff?text=Ghana+Economy',
        post_category: 'Business',
        post_source: 'Ghana Business News',
        date_added: new Date().toISOString(),
        views: 2156,
      },
      {
        id: 'fallback-2',
        post_title: 'Technology Sector Drives Innovation in West Africa',
        post_excerpt: 'West African tech startups are making significant strides in fintech, agriculture, and healthcare sectors.',
        post_content: 'Full article content here...',
        post_image: 'https://via.placeholder.com/400x200/2d72f0/ffffff?text=Tech+Innovation',
        post_category: 'Technology',
        post_source: 'Tech Africa',
        date_added: new Date().toISOString(),
        views: 1890,
      },
      {
        id: 'fallback-3',
        post_title: 'Sports: Local Teams Prepare for International Competitions',
        post_excerpt: 'Ghanaian sports teams are intensifying their preparations for upcoming international tournaments.',
        post_content: 'Full article content here...',
        post_image: 'https://via.placeholder.com/400x200/5a95ff/ffffff?text=Sports+News',
        post_category: 'Sports',
        post_source: 'Sports Ghana',
        date_added: new Date().toISOString(),
        views: 1247,
      }
    ];
  };

  const onRefresh = async () => {
    setIsRefreshing(true);
    await fetchNewsData(true); // Reset data on refresh
    setIsRefreshing(false);
  };

  const loadMoreData = async () => {
    if (!isLoadingMore && hasMoreData) {
      await fetchNewsData(false); // Load more data
    }
  };

  const handleCategoryPress = (categoryName: string) => {
    setSelectedCategory(categoryName);
    setSearchQuery(''); // Clear search when selecting category
  };

  const handleSearch = async () => {
    if (searchQuery.trim()) {
      setIsSearching(true);
      await fetchNewsData(true); // Reset data for search
      setIsSearching(false);
    }
  };

  const handleSearchClear = () => {
    setSearchQuery('');
    setSelectedCategory('For You');
  };

  const handleNewsPress = (article: NewsArticle) => {
    (navigation as any).navigate('DeepDive', { articleId: article.id });
  };

  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    const now = new Date();
    const diffTime = Math.abs(now.getTime() - date.getTime());
    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
    
    if (diffDays === 1) {
      return 'Today';
    } else if (diffDays === 2) {
      return 'Yesterday';
    } else if (diffDays <= 7) {
      return `${diffDays - 1} days ago`;
    } else {
      return date.toLocaleDateString('en-US', { 
        month: 'short', 
        day: 'numeric',
        year: date.getFullYear() !== now.getFullYear() ? 'numeric' : undefined
      });
    }
  };

  const renderNewsCard = ({ item }: { item: NewsArticle }) => (
    <TouchableOpacity
      style={dynamicStyles.newsCard}
      onPress={() => handleNewsPress(item)}
      activeOpacity={0.8}
    >
      <View style={dynamicStyles.newsCardImageContainer}>
        <Image
          source={{ uri: item.post_image || 'https://via.placeholder.com/400x200?text=No+Image' }}
          style={dynamicStyles.newsCardImage}
          resizeMode="cover"
          loadingIndicatorSource={{ uri: 'https://via.placeholder.com/400x200?text=Loading...' }}
          defaultSource={{ uri: 'https://via.placeholder.com/400x200?text=Loading...' }}
        />
        <View style={dynamicStyles.newsCardCategoryOverlay}>
          <Text style={dynamicStyles.newsCardCategoryText}>{item.post_category}</Text>
        </View>
      </View>
      
      <View style={dynamicStyles.newsCardContent}>
        <Text style={dynamicStyles.newsCardTitle} numberOfLines={2}>
          {item.post_title}
        </Text>
        
        <Text style={dynamicStyles.newsCardExcerpt} numberOfLines={2}>
          {item.post_excerpt}
        </Text>
        
        <View style={dynamicStyles.newsCardFooter}>
          <View style={dynamicStyles.newsCardDateContainer}>
            <Text style={dynamicStyles.newsCardDateIcon}>📅</Text>
            <Text style={dynamicStyles.newsCardDateText}>
              {formatDate(item.date_added)}
            </Text>
          </View>
          
          <TouchableOpacity 
            style={dynamicStyles.newsCardBookmark}
            onPress={() => {
              // TODO: Implement bookmark functionality
              Alert.alert('Bookmark', 'Bookmark functionality coming soon!');
            }}
            activeOpacity={0.7}
          >
            <Text style={dynamicStyles.newsCardBookmarkIcon}>🔖</Text>
          </TouchableOpacity>
        </View>
      </View>
    </TouchableOpacity>
  );

  const dynamicStyles = createDynamicStyles(isDark);

  return (
    <SafeAreaView style={dynamicStyles.container}>
      <StatusBar barStyle={isDark ? "light-content" : "dark-content"} backgroundColor={isDark ? "#1a1a1a" : "#ffffff"} />
      
      {/* Header */}
      <View style={dynamicStyles.header}>
        <TouchableOpacity
          style={dynamicStyles.backButton}
          onPress={() => navigation.goBack()}
          activeOpacity={0.7}
        >
          <Text style={dynamicStyles.backIcon}>←</Text>
        </TouchableOpacity>
        
        <TouchableOpacity 
          style={dynamicStyles.searchContainer}
          onPress={() => {
            // For now, show an alert. In the future, this could open a search modal
            Alert.alert(
              'Search',
              'Search functionality is available through category filtering. Select a category to see related news.',
              [{ text: 'OK' }]
            );
          }}
          activeOpacity={0.7}
        >
          <Text style={dynamicStyles.searchIcon}>🔍</Text>
          <Text style={dynamicStyles.searchText}>Search news...</Text>
        </TouchableOpacity>
        
        <TouchableOpacity
          style={dynamicStyles.heartButton}
          onPress={() => {}}
          activeOpacity={0.7}
        >
          <Text style={dynamicStyles.heartIcon}>♡</Text>
        </TouchableOpacity>
      </View>

      {/* Category Tabs */}
      <View style={dynamicStyles.categoryTabs}>
        <ScrollView 
          horizontal 
          showsHorizontalScrollIndicator={false}
          contentContainerStyle={dynamicStyles.categoryScrollContent}
        >
          {categories.map((category) => (
            <TouchableOpacity
              key={category.id}
              style={[
                dynamicStyles.categoryTab,
                category.isActive && dynamicStyles.categoryTabActive
              ]}
              onPress={() => handleCategoryPress(category.name)}
              activeOpacity={0.7}
            >
              <Text style={[
                dynamicStyles.categoryTabText,
                category.isActive && dynamicStyles.categoryTabTextActive
              ]}>
                {category.name}
              </Text>
            </TouchableOpacity>
          ))}
        </ScrollView>
      </View>

      {/* News Feed */}
      <View style={dynamicStyles.newsFeed}>
        {isLoading ? (
          <View style={dynamicStyles.loadingContainer}>
            <Text style={dynamicStyles.loadingText}>
              {searchQuery ? 'Searching news...' : `Loading ${selectedCategory.toLowerCase()} news...`}
            </Text>
          </View>
        ) : newsData.length === 0 ? (
          <View style={dynamicStyles.loadingContainer}>
            <Text style={dynamicStyles.loadingText}>
              No news found for {selectedCategory.toLowerCase()}
            </Text>
          </View>
        ) : (
          <FlatList
            data={newsData}
            renderItem={renderNewsCard}
            keyExtractor={(item) => item.id}
            showsVerticalScrollIndicator={false}
            refreshControl={
              <RefreshControl
                refreshing={isRefreshing}
                onRefresh={onRefresh}
                tintColor={isDark ? '#ffffff' : '#666'}
              />
            }
            onEndReached={loadMoreData}
            onEndReachedThreshold={0.5}
            ListFooterComponent={() => (
              isLoadingMore ? (
                <View style={dynamicStyles.loadingFooter}>
                  <Text style={dynamicStyles.loadingFooterText}>Loading more...</Text>
                </View>
              ) : null
            )}
            contentContainerStyle={dynamicStyles.newsListContent}
            // Performance optimizations
            removeClippedSubviews={true}
            maxToRenderPerBatch={5}
            windowSize={10}
            initialNumToRender={5}
            getItemLayout={(data, index) => ({
              length: 280, // Approximate height of news card
              offset: 280 * index,
              index,
            })}
          />
        )}
      </View>

      {/* Bottom CTA */}
      <View style={dynamicStyles.bottomCTA}>
        <Text style={dynamicStyles.bottomCTAText}>Feed your curiosity</Text>
      </View>
    </SafeAreaView>
  );
};

const createDynamicStyles = (isDark: boolean) => StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: isDark ? '#1a1a1a' : '#ffffff',
  },
  header: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 20,
    paddingVertical: 16,
    borderBottomWidth: 1,
    borderBottomColor: isDark ? '#404040' : '#f0f0f0',
  },
  backButton: {
    width: 40,
    height: 40,
    alignItems: 'center',
    justifyContent: 'center',
  },
  backIcon: {
    fontSize: 20,
    color: isDark ? '#ffffff' : '#1a1a1a',
  },
  searchContainer: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: isDark ? '#2d2d2d' : '#f8f9fa',
    borderRadius: 20,
    paddingHorizontal: 16,
    paddingVertical: 8,
    marginHorizontal: 16,
  },
  searchIcon: {
    fontSize: 16,
    color: isDark ? '#b3b3b3' : '#666',
    marginRight: 8,
  },
  searchText: {
    fontSize: 16,
    color: isDark ? '#b3b3b3' : '#666',
  },
  heartButton: {
    width: 40,
    height: 40,
    alignItems: 'center',
    justifyContent: 'center',
  },
  heartIcon: {
    fontSize: 20,
    color: isDark ? '#ffffff' : '#1a1a1a',
  },
  categoryTabs: {
    backgroundColor: isDark ? '#1a1a1a' : '#ffffff',
    borderBottomWidth: 1,
    borderBottomColor: isDark ? '#404040' : '#f0f0f0',
  },
  categoryScrollContent: {
    paddingHorizontal: 20,
    gap: 16,
  },
  categoryTab: {
    paddingHorizontal: 16,
    paddingVertical: 12,
    borderRadius: 20,
    backgroundColor: 'transparent',
  },
  categoryTabActive: {
    backgroundColor: isDark ? '#404040' : '#f8f9fa',
  },
  categoryTabText: {
    fontSize: 16,
    fontWeight: '500',
    color: isDark ? '#b3b3b3' : '#666',
  },
  categoryTabTextActive: {
    color: isDark ? '#ffffff' : '#1a1a1a',
    fontWeight: '600',
  },
  newsFeed: {
    flex: 1,
  },
  loadingContainer: {
    flex: 1,
    alignItems: 'center',
    justifyContent: 'center',
  },
  loadingText: {
    fontSize: 16,
    color: isDark ? '#b3b3b3' : '#666',
  },
  loadingFooter: {
    paddingVertical: 20,
    alignItems: 'center',
  },
  loadingFooterText: {
    fontSize: 14,
    color: isDark ? '#b3b3b3' : '#666',
  },
  newsListContent: {
    padding: 20,
    paddingBottom: 100,
  },
  newsCard: {
    backgroundColor: isDark ? '#2d2d2d' : '#ffffff',
    borderRadius: 16,
    marginBottom: 20,
    overflow: 'hidden',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: isDark ? 0.3 : 0.1,
    shadowRadius: 12,
    elevation: 6,
    borderWidth: 1,
    borderColor: isDark ? '#404040' : '#f0f0f0',
  },
  newsCardImageContainer: {
    position: 'relative',
    height: 200,
    overflow: 'hidden',
  },
  newsCardImage: {
    width: '100%',
    height: '100%',
  },
  newsCardCategoryOverlay: {
    position: 'absolute',
    top: 12,
    left: 12,
    backgroundColor: 'rgba(0, 0, 0, 0.7)',
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 12,
  },
  newsCardCategoryText: {
    color: '#ffffff',
    fontSize: 12,
    fontWeight: '600',
  },
  newsCardContent: {
    padding: 20,
  },
  newsCardTitle: {
    fontSize: 18,
    fontWeight: '700',
    color: isDark ? '#ffffff' : '#1a1a1a',
    marginBottom: 12,
    lineHeight: 26,
  },
  newsCardExcerpt: {
    fontSize: 14,
    color: isDark ? '#b3b3b3' : '#666',
    lineHeight: 20,
    marginBottom: 12,
  },
  newsCardFooter: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginTop: 8,
  },
  newsCardDateContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: 6,
  },
  newsCardDateIcon: {
    fontSize: 12,
  },
  newsCardDateText: {
    fontSize: 12,
    color: isDark ? '#b3b3b3' : '#666',
    fontWeight: '500',
  },
  newsCardBookmark: {
    padding: 4,
    borderRadius: 6,
    backgroundColor: isDark ? '#404040' : '#f8f9fa',
  },
  newsCardBookmarkIcon: {
    fontSize: 16,
    color: isDark ? '#ffffff' : '#666',
  },
  bottomCTA: {
    position: 'absolute',
    bottom: 40,
    left: 20,
    right: 20,
    alignItems: 'center',
  },
  bottomCTAText: {
    fontSize: 24,
    fontWeight: '700',
    color: isDark ? '#ffffff' : '#1a1a1a',
    textAlign: 'center',
  },
});

export default DiscoverScreen;
