import React, { useEffect } from 'react';
import { View, StyleSheet, FlatList, RefreshControl, TouchableOpacity } from 'react-native';
import { useDispatch, useSelector } from 'react-redux';
import { Text, ActivityIndicator, Surface } from 'react-native-paper';
import { RootState, AppDispatch } from '../store';
import { fetchLatestNews, fetchNewsByCategory, setSelectedCategory } from '../store/slices/newsSlice';
import { NewsArticle } from '../types';

const HomeScreen: React.FC = () => {
  const dispatch = useDispatch<AppDispatch>();
  const { articles, categories, isLoading, selectedCategory } = useSelector((state: RootState) => state.news);

  useEffect(() => {
    dispatch(fetchLatestNews(20));
  }, [dispatch]);

  const handleRefresh = () => {
    if (selectedCategory) {
      dispatch(fetchNewsByCategory({ category: selectedCategory }));
    } else {
      dispatch(fetchLatestNews(20));
    }
  };

  const handleCategoryPress = (category: string) => {
    if (selectedCategory === category) {
      dispatch(setSelectedCategory(null));
      dispatch(fetchLatestNews(20));
    } else {
      dispatch(setSelectedCategory(category));
      dispatch(fetchNewsByCategory({ category }));
    }
  };

  const renderArticle = ({ item }: { item: NewsArticle }) => (
    <TouchableOpacity style={styles.articleCard} activeOpacity={0.7}>
      <View style={styles.articleImageContainer}>
        <View style={[styles.articleImage, { backgroundColor: '#f0f0f0' }]} />
        <View style={styles.categoryBadge}>
          <Text style={styles.categoryText}>{item.post_category}</Text>
        </View>
      </View>
      <View style={styles.articleContent}>
        <Text style={styles.articleTitle} numberOfLines={2}>
          {item.post_title}
        </Text>
        <Text style={styles.articleExcerpt} numberOfLines={2}>
          {item.post_excerpt}
        </Text>
        <View style={styles.articleMeta}>
          <Text style={styles.sourceText}>{item.post_source}</Text>
          <Text style={styles.viewsText}>{item.views} views</Text>
        </View>
      </View>
    </TouchableOpacity>
  );

  const renderCategory = (category: string) => (
    <TouchableOpacity
      key={category}
      style={[
        styles.categoryButton,
        selectedCategory === category && styles.selectedCategoryButton
      ]}
      onPress={() => handleCategoryPress(category)}
      activeOpacity={0.7}
    >
      <Text style={[
        styles.categoryButtonText,
        selectedCategory === category && styles.selectedCategoryButtonText
      ]}>
        {category}
      </Text>
    </TouchableOpacity>
  );

  if (isLoading && articles.length === 0) {
    return (
      <View style={styles.loadingContainer}>
        <ActivityIndicator size="large" color="#0a57d4" />
      </View>
    );
  }

  return (
    <View style={styles.container}>
      {/* Category Filter */}
      <View style={styles.categoriesContainer}>
        <FlatList
          horizontal
          showsHorizontalScrollIndicator={false}
          data={categories}
          renderItem={({ item }) => renderCategory(item)}
          keyExtractor={(item) => item}
          contentContainerStyle={styles.categoriesList}
        />
      </View>

      {/* News Articles */}
      <FlatList
        data={articles}
        renderItem={renderArticle}
        keyExtractor={(item) => item.id}
        refreshControl={
          <RefreshControl
            refreshing={isLoading}
            onRefresh={handleRefresh}
            colors={['#0a57d4']}
            tintColor="#0a57d4"
          />
        }
        contentContainerStyle={styles.articlesList}
        showsVerticalScrollIndicator={false}
      />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fafafa',
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  categoriesContainer: {
    backgroundColor: '#fff',
    paddingVertical: 16,
    paddingHorizontal: 20,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
  },
  categoriesList: {
    paddingHorizontal: 0,
  },
  categoryButton: {
    paddingHorizontal: 16,
    paddingVertical: 8,
    marginRight: 12,
    borderRadius: 20,
    backgroundColor: '#f5f5f5',
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  selectedCategoryButton: {
    backgroundColor: '#0a57d4',
    borderColor: '#0a57d4',
  },
  categoryButtonText: {
    fontSize: 14,
    color: '#666',
    fontWeight: '500',
  },
  selectedCategoryButtonText: {
    color: '#fff',
  },
  articlesList: {
    padding: 20,
  },
  articleCard: {
    backgroundColor: '#fff',
    borderRadius: 12,
    marginBottom: 20,
    overflow: 'hidden',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 8,
    elevation: 3,
  },
  articleImageContainer: {
    position: 'relative',
    height: 200,
  },
  articleImage: {
    width: '100%',
    height: '100%',
  },
  categoryBadge: {
    position: 'absolute',
    top: 12,
    left: 12,
    backgroundColor: '#0a57d4',
    paddingHorizontal: 8,
    paddingVertical: 4,
    borderRadius: 6,
  },
  categoryText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: '600',
  },
  articleContent: {
    padding: 16,
  },
  articleTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#333',
    lineHeight: 24,
    marginBottom: 8,
  },
  articleExcerpt: {
    fontSize: 14,
    color: '#666',
    lineHeight: 20,
    marginBottom: 12,
  },
  articleMeta: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  sourceText: {
    fontSize: 12,
    color: '#999',
    fontWeight: '500',
  },
  viewsText: {
    fontSize: 12,
    color: '#999',
  },
});

export default HomeScreen;
