import React, { useState } from 'react';
import {
  View,
  StyleSheet,
  ScrollView,
  TouchableOpacity,
  Text,
  SafeAreaView,
  StatusBar,
  Alert,
} from 'react-native';
import { useNavigation } from '@react-navigation/native';
import { useTheme } from '../contexts/ThemeContext';

interface SettingItem {
  id: string;
  title: string;
  subtitle?: string;
  icon: string;
  hasToggle?: boolean;
  toggleValue?: boolean;
  onToggle?: () => void;
  onPress?: () => void;
}

const ProfileScreen: React.FC = () => {
  const navigation = useNavigation();
  const { isDark, toggleTheme } = useTheme();
  const [notificationsEnabled, setNotificationsEnabled] = useState(true);

  const handleNotificationToggle = () => {
    setNotificationsEnabled(!notificationsEnabled);
    Alert.alert(
      'Notifications',
      notificationsEnabled ? 'Push notifications disabled' : 'Push notifications enabled',
      [{ text: 'OK' }]
    );
  };

  const handleSettingPress = (settingId: string) => {
    switch (settingId) {
      case 'language':
        Alert.alert('Language', 'Language settings coming soon!');
        break;
      case 'categories':
        Alert.alert('News Categories', 'Category preferences coming soon!');
        break;
      case 'location':
        Alert.alert('Location', 'Location settings coming soon!');
        break;
      case 'data':
        Alert.alert('Data Usage', 'Data usage settings coming soon!');
        break;
      case 'privacy':
        Alert.alert('Privacy & Security', 'Privacy settings coming soon!');
        break;
      case 'about':
        Alert.alert(
          'About Mews',
          'Mews AI v1.0.0\n\nAI-powered news discovery and analysis.\n\nMade with ❤️ in Ghana',
          [{ text: 'OK' }]
        );
        break;
      default:
        break;
    }
  };

  const settings: SettingItem[] = [
    {
      id: 'theme',
      title: 'Dark Mode',
      subtitle: isDark ? 'Dark theme enabled' : 'Light theme enabled',
      icon: isDark ? '🌙' : '☀️',
      hasToggle: true,
      toggleValue: isDark,
      onToggle: toggleTheme,
    },
    {
      id: 'notifications',
      title: 'Push Notifications',
      subtitle: 'Get notified about breaking news',
      icon: '🔔',
      hasToggle: true,
      toggleValue: notificationsEnabled,
      onToggle: handleNotificationToggle,
    },
    {
      id: 'language',
      title: 'App Language',
      subtitle: 'English',
      icon: '🌍',
      onPress: () => handleSettingPress('language'),
    },
    {
      id: 'categories',
      title: 'News Categories',
      subtitle: 'Customize your news feed',
      icon: '📰',
      onPress: () => handleSettingPress('categories'),
    },
    {
      id: 'location',
      title: 'Location',
      subtitle: 'Ghana',
      icon: '📍',
      onPress: () => handleSettingPress('location'),
    },
    {
      id: 'data',
      title: 'Data Usage',
      subtitle: 'Manage your data consumption',
      icon: '📊',
      onPress: () => handleSettingPress('data'),
    },
    {
      id: 'privacy',
      title: 'Privacy & Security',
      subtitle: 'Manage your privacy settings',
      icon: '🔒',
      onPress: () => handleSettingPress('privacy'),
    },
    {
      id: 'about',
      title: 'About Mews',
      subtitle: 'Version 1.0.0',
      icon: 'ℹ️',
      onPress: () => handleSettingPress('about'),
    },
  ];

  const dynamicStyles = createDynamicStyles(isDark);

  return (
    <SafeAreaView style={dynamicStyles.container}>
      <StatusBar barStyle={isDark ? "light-content" : "dark-content"} backgroundColor={isDark ? "#1a1a1a" : "#ffffff"} />
      
      {/* Header */}
      <View style={dynamicStyles.header}>
        <TouchableOpacity
          style={dynamicStyles.closeButton}
          onPress={() => navigation.goBack()}
          activeOpacity={0.7}
        >
          <Text style={dynamicStyles.closeIcon}>×</Text>
        </TouchableOpacity>
        <Text style={dynamicStyles.headerTitle}>Settings</Text>
      </View>

      <ScrollView style={dynamicStyles.content} showsVerticalScrollIndicator={false}>
        {/* User Profile Section */}
        <View style={dynamicStyles.profileSection}>
          <View style={dynamicStyles.profileAvatar}>
            <Text style={dynamicStyles.profileAvatarText}>M</Text>
          </View>
          <View style={dynamicStyles.profileInfo}>
            <Text style={dynamicStyles.profileName}>Mews User</Text>
            <TouchableOpacity 
              style={dynamicStyles.manageAccountButton}
              onPress={() => Alert.alert('Account', 'Account management coming soon!')}
              activeOpacity={0.7}
            >
              <Text style={dynamicStyles.manageAccountText}>Manage Account</Text>
            </TouchableOpacity>
          </View>
        </View>

        {/* Settings List */}
        <View style={dynamicStyles.settingsList}>
          {settings.map((setting) => (
            <TouchableOpacity
              key={setting.id}
              style={dynamicStyles.settingItem}
              onPress={setting.onPress}
              activeOpacity={0.7}
              disabled={!setting.onPress}
            >
              <View style={dynamicStyles.settingLeft}>
                <Text style={dynamicStyles.settingIcon}>{setting.icon}</Text>
                <View style={dynamicStyles.settingTextContainer}>
                  <Text style={dynamicStyles.settingTitle}>{setting.title}</Text>
                  {setting.subtitle && (
                    <Text style={dynamicStyles.settingSubtitle}>{setting.subtitle}</Text>
                  )}
                </View>
              </View>
              
              <View style={dynamicStyles.settingRight}>
                {setting.hasToggle ? (
                  <TouchableOpacity
                    style={[
                      dynamicStyles.toggle,
                      setting.toggleValue && dynamicStyles.toggleActive
                    ]}
                    onPress={setting.onToggle}
                    activeOpacity={0.8}
                  >
                    <View style={[
                      dynamicStyles.toggleThumb,
                      setting.toggleValue && dynamicStyles.toggleThumbActive
                    ]} />
                  </TouchableOpacity>
                ) : (
                  <Text style={dynamicStyles.arrowIcon}>›</Text>
                )}
              </View>
            </TouchableOpacity>
          ))}
        </View>

        {/* Footer */}
        <View style={dynamicStyles.footer}>
          <Text style={dynamicStyles.footerText}>Made with ❤️ in Ghana</Text>
          <Text style={dynamicStyles.footerText}>© 2024 Mews AI</Text>
        </View>
      </ScrollView>
    </SafeAreaView>
  );
};

const createDynamicStyles = (isDark: boolean) => StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: isDark ? '#1a1a1a' : '#ffffff',
  },
  header: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 20,
    paddingVertical: 16,
    borderBottomWidth: 1,
    borderBottomColor: isDark ? '#404040' : '#f0f0f0',
    position: 'relative',
  },
  closeButton: {
    position: 'absolute',
    left: 20,
    width: 32,
    height: 32,
    alignItems: 'center',
    justifyContent: 'center',
  },
  closeIcon: {
    fontSize: 24,
    color: isDark ? '#ffffff' : '#1a1a1a',
    fontWeight: '300',
  },
  headerTitle: {
    fontSize: 20,
    fontWeight: '600',
    color: isDark ? '#ffffff' : '#1a1a1a',
  },
  content: {
    flex: 1,
  },
  profileSection: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: 20,
    paddingVertical: 24,
    borderBottomWidth: 1,
    borderBottomColor: isDark ? '#404040' : '#f0f0f0',
  },
  profileAvatar: {
    width: 60,
    height: 60,
    borderRadius: 30,
    backgroundColor: '#0a57d4',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 16,
  },
  profileAvatarText: {
    color: '#ffffff',
    fontSize: 24,
    fontWeight: '600',
  },
  profileInfo: {
    flex: 1,
  },
  profileName: {
    fontSize: 20,
    fontWeight: '600',
    color: isDark ? '#ffffff' : '#1a1a1a',
    marginBottom: 4,
  },
  manageAccountButton: {
    alignSelf: 'flex-start',
  },
  manageAccountText: {
    fontSize: 16,
    color: '#0a57d4',
    fontWeight: '500',
  },
  settingsList: {
    paddingVertical: 8,
  },
  settingItem: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 20,
    paddingVertical: 16,
    borderBottomWidth: 1,
    borderBottomColor: isDark ? '#404040' : '#f0f0f0',
  },
  settingLeft: {
    flexDirection: 'row',
    alignItems: 'center',
    flex: 1,
  },
  settingIcon: {
    fontSize: 20,
    marginRight: 16,
    width: 24,
    textAlign: 'center',
  },
  settingTextContainer: {
    flex: 1,
  },
  settingTitle: {
    fontSize: 16,
    fontWeight: '500',
    color: isDark ? '#ffffff' : '#1a1a1a',
    marginBottom: 2,
  },
  settingSubtitle: {
    fontSize: 14,
    color: isDark ? '#b3b3b3' : '#666',
  },
  settingRight: {
    alignItems: 'center',
    justifyContent: 'center',
  },
  arrowIcon: {
    fontSize: 18,
    color: isDark ? '#b3b3b3' : '#666',
    fontWeight: '300',
  },
  toggle: {
    width: 50,
    height: 30,
    borderRadius: 15,
    backgroundColor: isDark ? '#404040' : '#e9ecef',
    padding: 2,
    justifyContent: 'center',
    transition: 'all 0.2s ease-in-out',
  },
  toggleActive: {
    backgroundColor: '#0a57d4',
  },
  toggleThumb: {
    width: 26,
    height: 26,
    borderRadius: 13,
    backgroundColor: '#ffffff',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 2,
    transition: 'transform 0.2s ease-in-out',
  },
  toggleThumbActive: {
    transform: [{ translateX: 20 }],
  },
  footer: {
    alignItems: 'center',
    paddingVertical: 32,
    paddingHorizontal: 20,
  },
  footerText: {
    fontSize: 14,
    color: isDark ? '#b3b3b3' : '#666',
    marginBottom: 4,
  },
});

export default ProfileScreen;