<?php

namespace App\Console\Commands;

use App\Models\Election\Election;
use App\Models\Election\Party;
use App\Models\Election\Region;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class ClearElectionData extends Command
{
    protected $signature = 'election:clear
                            {--keep-structure : Only clear result/scraped data; keep elections and reference data}
                            {--force : Skip confirmation}';

    protected $description = 'Clear all election data and optionally seed Peace FM–only elections (parties, regions, 8 presidential elections)';

    public function handle(): int
    {
        if (! $this->option('force') && ! $this->confirm('This will delete all election data. Continue?')) {
            return self::FAILURE;
        }

        $keepStructure = $this->option('keep-structure');

        $this->info('Clearing election-related data...');

        // Order matters: child tables first (foreign keys)
        $tables = [
            'election_chat_logs',
            'election_bookmarks',
            'poll_results',
            'polls',
            'ai_predictions',
            'social_mentions',
            'parliamentary_seats',
            'election_results_constituency',
            'election_results_regional',
            'election_results_national',
            'election_scraped_data',
            'election_candidates',
        ];

        foreach ($tables as $table) {
            if (\Illuminate\Support\Facades\Schema::hasTable($table)) {
                DB::table($table)->delete();
                $this->line("  Cleared: {$table}");
            }
        }

        if (! $keepStructure) {
            if (\Illuminate\Support\Facades\Schema::hasTable('elections')) {
                DB::table('elections')->delete();
                $this->line('  Cleared: elections');
            }
        }

        $this->info('Seeding Peace FM reference data (parties, regions, elections)...');
        $this->seedPeaceFmReferenceData();

        $this->info('Done. Election data is cleared. Use Peace FM scrapers and election:import-peacefm to populate.');
        return self::SUCCESS;
    }

    private function seedPeaceFmReferenceData(): void
    {
        // Ensure NPP and NDC exist
        $parties = [
            ['name' => 'New Patriotic Party', 'short_name' => 'NPP', 'color' => '#0057B7'],
            ['name' => 'National Democratic Congress', 'short_name' => 'NDC', 'color' => '#009739'],
        ];
        foreach ($parties as $p) {
            Party::firstOrCreate(
                ['short_name' => $p['short_name']],
                array_merge($p, ['is_active' => true])
            );
        }
        $this->line('  Parties: NPP, NDC');

        // 16 Ghana regions (codes match common usage)
        $regions = [
            ['name' => 'Ahafo', 'code' => 'AH'],
            ['name' => 'Ashanti', 'code' => 'AS'],
            ['name' => 'Bono', 'code' => 'BO'],
            ['name' => 'Bono East', 'code' => 'BE'],
            ['name' => 'Central', 'code' => 'CP'],
            ['name' => 'Eastern', 'code' => 'EP'],
            ['name' => 'Greater Accra', 'code' => 'GA'],
            ['name' => 'North East', 'code' => 'NE'],
            ['name' => 'Northern', 'code' => 'NP'],
            ['name' => 'Oti', 'code' => 'OT'],
            ['name' => 'Savannah', 'code' => 'SV'],
            ['name' => 'Upper East', 'code' => 'UE'],
            ['name' => 'Upper West', 'code' => 'UW'],
            ['name' => 'Volta', 'code' => 'TV'],
            ['name' => 'Western', 'code' => 'WP'],
            ['name' => 'Western North', 'code' => 'WN'],
        ];
        foreach ($regions as $r) {
            Region::firstOrCreate(
                ['code' => $r['code']],
                array_merge($r, [])
            );
        }
        $this->line('  Regions: 16');

        // Peace FM presidential elections (and runoffs)
        $elections = [
            ['year' => 2000, 'type' => 'presidential', 'name' => '2000 Presidential Election'],
            ['year' => 2000, 'type' => 'presidential_runoff', 'name' => '2000 Presidential Runoff'],
            ['year' => 2004, 'type' => 'presidential', 'name' => '2004 Presidential Election'],
            ['year' => 2008, 'type' => 'presidential', 'name' => '2008 Presidential Election'],
            ['year' => 2008, 'type' => 'presidential_runoff', 'name' => '2008 Presidential Runoff'],
            ['year' => 2016, 'type' => 'presidential', 'name' => '2016 Presidential Election'],
            ['year' => 2020, 'type' => 'presidential', 'name' => '2020 Presidential Election'],
            ['year' => 2024, 'type' => 'presidential', 'name' => '2024 Presidential Election'],
        ];
        foreach ($elections as $e) {
            Election::firstOrCreate(
                ['year' => $e['year'], 'type' => $e['type']],
                array_merge($e, ['status' => 'completed'])
            );
        }
        $this->line('  Elections: 8 (2000, 2000 Runoff, 2004, 2008, 2008 Runoff, 2016, 2020, 2024)');
    }
}
