<?php

namespace App\Console\Commands;

use App\Models\Election\Election;
use App\Models\Election\ElectionScrapedData;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class Import3News2020 extends Command
{
    protected $signature = 'election:import-3news {file=storage/app/3news_2020_presidential.json : Path to JSON from scripts/scrape-3news-2020.mjs}';
    protected $description = 'Import 2020 presidential data from 3 News JSON (run npm run scrape:3news first)';

    public function handle(): int
    {
        $path = base_path($this->argument('file'));
        if (!is_file($path)) {
            $this->error("File not found: {$path}");
            $this->line('Run: npm run scrape:3news (then npx playwright install chromium if needed)');
            return self::FAILURE;
        }

        $json = json_decode(file_get_contents($path), true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->error('Invalid JSON: ' . json_last_error_msg());
            return self::FAILURE;
        }

        $election = Election::where('year', 2020)->where('type', 'presidential')->first();
        if (!$election) {
            $this->error('2020 presidential election not found. Run ElectionDataSeeder first.');
            return self::FAILURE;
        }

        $payload = [
            'source_url' => $json['source_url'] ?? null,
            'scraped_at' => $json['scraped_at'] ?? now()->toIso8601String(),
            'regions' => $json['regions'] ?? [],
            'constituencies' => $json['constituencies'] ?? [],
            'candidate_constituency_wins' => $json['candidate_constituency_wins'] ?? [],
            'regional_npp_pct_2020' => $json['regional_npp_pct_2020'] ?? [],
            'raw_tables' => $json['raw_tables'] ?? [],
        ];

        ElectionScrapedData::updateOrCreate(
            [
                'election_id' => $election->id,
                'data_type' => '3news_presidential_2020',
            ],
            [
                'payload' => $payload,
                'source_url' => $payload['source_url'],
                'scraped_at' => now(),
            ]
        );

        $this->info('Imported 3 News 2020 data.');
        $this->line('Regions: ' . count($payload['regions']));
        $this->line('Constituencies: ' . count($payload['constituencies']));
        $this->line('Candidate wins: ' . count($payload['candidate_constituency_wins']));
        return self::SUCCESS;
    }
}
