<?php

namespace App\Console\Commands;

use App\Models\Election\Region;
use App\Models\Election\RegionCensus;
use Illuminate\Console\Command;

class ImportCensusData extends Command
{
    protected $signature = 'election:census-import {year : Census year (2000, 2010, or 2021)} {file : Path to JSON file with region populations}';
    protected $description = 'Import regional population from JSON. JSON format: [{"region":"Ashanti","population":1234567}, ...] or {"Ashanti":1234567,...}';

    public function handle(): int
    {
        $year = (int) $this->argument('year');
        if (! in_array($year, [2000, 2010, 2021], true)) {
            $this->error('Year must be 2000, 2010, or 2021.');
            return self::FAILURE;
        }

        $path = base_path($this->argument('file'));
        if (! is_file($path)) {
            $this->error("File not found: {$path}");
            return self::FAILURE;
        }

        $json = json_decode(file_get_contents($path), true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->error('Invalid JSON.');
            return self::FAILURE;
        }

        $regions = Region::all()->keyBy('name');
        $rows = $this->normalizeRows($json);
        $imported = 0;
        foreach ($rows as $regionName => $population) {
            $region = $regions->get($regionName);
            if (! $region) {
                $this->warn("Unknown region: {$regionName}");
                continue;
            }
            $pop = (int) $population;
            if ($pop <= 0) {
                continue;
            }
            RegionCensus::updateOrCreate(
                ['census_year' => $year, 'region_id' => $region->id],
                ['population' => $pop]
            );
            $imported++;
        }
        $this->info("Imported {$imported} regions for {$year} census.");
        return self::SUCCESS;
    }

    private function normalizeRows(array $json): array
    {
        if (isset($json[0]) && is_array($json[0])) {
            $out = [];
            foreach ($json as $row) {
                $name = $row['region'] ?? $row['name'] ?? $row['Region'] ?? null;
                $pop = $row['population'] ?? $row['Population'] ?? null;
                if ($name !== null) {
                    $out[(string) $name] = $pop;
                }
            }
            return $out;
        }
        return $json;
    }
}
