<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use PhpOffice\PhpSpreadsheet\IOFactory;

class ImportElectionData extends Command
{
    protected $signature = 'election:import-excel {file=election_history/Election Data.xlsx : Path to Election Data.xlsx}';
    protected $description = 'Import presidential/parliamentary election results from Election Data.xlsx (election_history folder)';

    /** Region name as in Excel -> our regions.name */
    private static $regionMap = [
        'Ahafo' => 'Ahafo',
        'Ashanti' => 'Ashanti',
        'BONO' => 'Bono',
        'Bono' => 'Bono',
        'Bono East' => 'Bono East',
        'BONO EAST' => 'Bono East',
        'Central' => 'Central',
        'Eastern' => 'Eastern',
        'Greater Accra' => 'Greater Accra',
        'North East' => 'North East',
        'Northern' => 'Northern',
        'Oti' => 'Oti',
        'Savannah' => 'Savannah',
        'Upper East' => 'Upper East',
        'Upper West' => 'Upper West',
        'Volta' => 'Volta',
        'Western' => 'Western',
        'Western North' => 'Western North',
    ];

    public function handle(): int
    {
        $path = base_path($this->argument('file'));
        if (!is_file($path)) {
            $this->error("File not found: {$path}");
            return self::FAILURE;
        }

        $this->info("Loading: {$path}");
        $spreadsheet = IOFactory::load($path);
        $sheetNames = $spreadsheet->getSheetNames();
        $this->info('Sheets: ' . implode(', ', $sheetNames));

        $nppId = DB::table('parties')->where('short_name', 'NPP')->value('id');
        $ndcId = DB::table('parties')->where('short_name', 'NDC')->value('id');
        $othersId = DB::table('parties')->where('short_name', 'OTH')->value('id');
        if (!$nppId || !$ndcId || !$othersId) {
            $this->error('Run ElectionDataSeeder first (parties/regions).');
            return self::FAILURE;
        }

        $imported = 0;
        $skipped = 0;
        $errors = [];

        foreach ($sheetNames as $sheetName) {
            $parsed = $this->parseSheetName($sheetName);
            if (!$parsed) {
                $this->warn("Skipping sheet (unknown format): {$sheetName}");
                $skipped++;
                continue;
            }
            [$year, $type] = $parsed;

            $electionId = DB::table('elections')
                ->where('year', $year)
                ->where('type', $type)
                ->value('id');
            if (!$electionId) {
                $this->warn("No election id for {$year} {$type}");
                continue;
            }

            $sheet = $spreadsheet->getSheetByName($sheetName);
            $rows = $sheet->toArray(null, true, true, true);
            $rows = array_values($rows);
            $header = array_map('trim', array_values($rows[0] ?? []));
            $colIdx = $this->columnIndices($header);
            if (!$colIdx) {
                $errors[] = "Sheet {$sheetName}: could not find REGION, NPP, NDC columns";
                continue;
            }

            $candidates = $this->candidatesForElection($year, $type);
            $nppCandidateId = $candidates['npp'] ? DB::table('candidates')->where('name', $candidates['npp'])->value('id') : null;
            $ndcCandidateId = $candidates['ndc'] ? DB::table('candidates')->where('name', $candidates['ndc'])->value('id') : null;

            foreach (array_slice($rows, 1) as $rowNum => $row) {
                $row = array_values(is_array($row) ? $row : iterator_to_array($row));
                $regionName = trim((string) ($row[$colIdx['region']] ?? ''));
                if ($regionName === '') {
                    continue;
                }
                $regionNorm = self::$regionMap[$regionName] ?? $regionName;
                $regionId = DB::table('regions')->where('name', $regionNorm)->value('id');
                if (!$regionId) {
                    $errors[] = "Sheet {$sheetName} row " . ($rowNum + 2) . ": unknown region '{$regionName}'";
                    continue;
                }

                $npp = $this->parseVoteCell($row[$colIdx['npp']] ?? '');
                $ndc = $this->parseVoteCell($row[$colIdx['ndc']] ?? '');
                $others = $this->parseVoteCell($row[$colIdx['others']] ?? '');
                if ($npp === null && $ndc === null && $others === null) {
                    continue;
                }
                $nppV = $npp['votes'] ?? 0;
                $ndcV = $ndc['votes'] ?? 0;
                $othV = $others['votes'] ?? 0;
                $total = $nppV + $ndcV + $othV;
                if ($total <= 0) {
                    continue;
                }
                $nppPct = $npp['pct'] ?? round(100 * $nppV / $total, 2);
                $ndcPct = $ndc['pct'] ?? round(100 * $ndcV / $total, 2);
                $othPct = $others['pct'] ?? round(100 * $othV / $total, 2);
                $winner = ($nppV >= $ndcV && $nppV >= $othV) ? 'NPP' : (($ndcV >= $othV) ? 'NDC' : 'OTH');

                DB::table('election_results_regional')->updateOrInsert(
                    [
                        'election_id' => $electionId,
                        'region_id' => $regionId,
                        'party_id' => $nppId,
                    ],
                    [
                        'candidate_id' => $nppCandidateId,
                        'votes' => $nppV,
                        'percentage' => $nppPct,
                        'total_votes_in_region' => $total,
                        'is_region_winner' => $winner === 'NPP',
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]
                );
                DB::table('election_results_regional')->updateOrInsert(
                    [
                        'election_id' => $electionId,
                        'region_id' => $regionId,
                        'party_id' => $ndcId,
                    ],
                    [
                        'candidate_id' => $ndcCandidateId,
                        'votes' => $ndcV,
                        'percentage' => $ndcPct,
                        'total_votes_in_region' => $total,
                        'is_region_winner' => $winner === 'NDC',
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]
                );
                if ($othV > 0) {
                    DB::table('election_results_regional')->updateOrInsert(
                        [
                            'election_id' => $electionId,
                            'region_id' => $regionId,
                            'party_id' => $othersId,
                        ],
                        [
                            'candidate_id' => null,
                            'votes' => $othV,
                            'percentage' => $othPct,
                            'total_votes_in_region' => $total,
                            'is_region_winner' => false,
                            'created_at' => now(),
                            'updated_at' => now(),
                        ]
                    );
                }
                $imported++;
            }
        }

        foreach ($errors as $err) {
            $this->error($err);
        }
        $this->info("Imported: {$imported} regional result rows. Skipped sheets: {$skipped}.");
        return empty($errors) ? self::SUCCESS : self::FAILURE;
    }

    private function parseSheetName(string $name): ?array
    {
        $name = trim($name);
        // Sheets: "2000", "2000 RUN.OFF", "2004", "2008", "2008 RUN.OFF", "2012", "2016", "2020"
        if (preg_match('/^(\d{4})\s*(?:RUN\.?OFF|Runoff|runoff)?$/i', $name, $m)) {
            $year = (int) $m[1];
            $type = (preg_match('/RUN\.?OFF|runoff/i', $name)) ? 'presidential_runoff' : 'presidential';
            return [$year, $type];
        }
        if (preg_match('/^(\d{4})\s*(?:Parliamentary|parliamentary)$/i', $name, $m)) {
            return [(int) $m[1], 'parliamentary'];
        }
        return null;
    }

    private function columnIndices(array $header): ?array
    {
        $idx = ['region' => null, 'npp' => null, 'ndc' => null, 'others' => null];
        foreach ($header as $i => $h) {
            $h = strtoupper($h);
            if ($h === 'REGION') {
                $idx['region'] = $i;
            } elseif ($h === 'NPP') {
                $idx['npp'] = $i;
            } elseif ($h === 'NDC') {
                $idx['ndc'] = $i;
            } elseif ($h === 'OTHERS' || $h === 'OTHER') {
                $idx['others'] = $i;
            }
        }
        if ($idx['region'] === null || $idx['npp'] === null || $idx['ndc'] === null) {
            return null;
        }
        if ($idx['others'] === null) {
            $idx['others'] = max(array_filter($idx)) + 1;
        }
        return $idx;
    }

    /** Parse "77,779 (56.69%)" or "77779" -> ['votes' => 77779, 'pct' => 56.69] */
    private function parseVoteCell($cell): ?array
    {
        $s = is_numeric($cell) ? (string) (int) $cell : trim((string) $cell);
        if ($s === '') {
            return ['votes' => 0, 'pct' => 0.0];
        }
        $votes = null;
        $pct = null;
        if (preg_match('/^([\d,\s]+)\s*\(([\d.]+)\s*%?\)/', $s, $m)) {
            $votes = (int) str_replace([',', ' '], '', $m[1]);
            $pct = (float) $m[2];
        } elseif (preg_match('/^([\d,]+)/', $s, $m)) {
            $votes = (int) str_replace(',', '', $m[1]);
            if (preg_match('/\(([\d.]+)\s*%?\)/', $s, $m2)) {
                $pct = (float) $m2[1];
            }
        }
        if ($votes === null && $pct === null && is_numeric($s)) {
            $votes = (int) str_replace(',', '', $s);
        }
        if ($votes === null && $pct === null) {
            return null;
        }
        return [
            'votes' => $votes ?? 0,
            'pct' => $pct ?? 0.0,
        ];
    }

    private function candidatesForElection(int $year, string $type): array
    {
        $candidates = [
            2000 => ['npp' => 'John Agyekum Kuffour', 'ndc' => 'John Atta Mills'],
            2004 => ['npp' => 'John Agyekum Kuffour', 'ndc' => 'John Atta Mills'],
            2008 => ['npp' => 'Nana Akufo-Addo', 'ndc' => 'John Atta Mills'],
            2012 => ['npp' => 'Nana Akufo-Addo', 'ndc' => 'John Dramani Mahama'],
            2016 => ['npp' => 'Nana Akufo-Addo', 'ndc' => 'John Dramani Mahama'],
            2020 => ['npp' => 'Nana Akufo-Addo', 'ndc' => 'John Dramani Mahama'],
        ];
        return $candidates[$year] ?? ['npp' => null, 'ndc' => null];
    }
}
