<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\OneSignalService;

class OneSignalProductionSetup extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'onesignal:production-setup';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Setup OneSignal for production deployment on mewsai.fumbo.ai';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🚀 OneSignal Production Setup for mewsai.fumbo.ai');
        $this->line('================================================');

        // Check configuration
        $this->info('📋 Checking OneSignal Configuration...');
        
        $appId = config('services.onesignal.app_id');
        $restApiKey = config('services.onesignal.rest_api_key');
        
        if (!$appId) {
            $this->error('❌ ONESIGNAL_APP_ID not set in .env');
            return 1;
        }
        
        if (!$restApiKey) {
            $this->error('❌ ONESIGNAL_REST_API_KEY not set in .env');
            return 1;
        }
        
        $this->info('✅ App ID: ' . $appId);
        $this->info('✅ REST API Key: Set');
        
        // Test OneSignal API connection
        $this->info('🔗 Testing OneSignal API Connection...');
        
        try {
            $oneSignalService = app(OneSignalService::class);
            
            // Test with a simple API call
            $result = $oneSignalService->makeRequest('apps/' . $appId, [], 'GET');
            
            if ($result) {
                $this->info('✅ OneSignal API connection successful');
                $this->info('✅ App Name: ' . ($result['name'] ?? 'Unknown'));
            } else {
                $this->error('❌ OneSignal API connection failed');
                return 1;
            }
        } catch (\Exception $e) {
            $this->error('❌ OneSignal API error: ' . $e->getMessage());
            return 1;
        }
        
        // Check database tables
        $this->info('🗄️ Checking Database Tables...');
        
        $tables = [
            'notification_campaigns',
            'user_notification_subscriptions',
            'notification_logs'
        ];
        
        foreach ($tables as $table) {
            try {
                \DB::table($table)->count();
                $this->info("✅ Table '{$table}' exists");
            } catch (\Exception $e) {
                $this->error("❌ Table '{$table}' missing - run migrations");
                return 1;
            }
        }
        
        // Production checklist
        $this->info('📝 Production Deployment Checklist:');
        $this->line('');
        
        $checklist = [
            'OneSignal dashboard configured for mewsai.fumbo.ai',
            'Environment variables set in production .env',
            'Test pages accessible on production domain',
            'Admin panel accessible and working',
            'User subscription flow tested',
            'Test notifications being delivered',
            'Analytics tracking properly configured'
        ];
        
        foreach ($checklist as $item) {
            $this->line("☐ {$item}");
        }
        
        $this->line('');
        $this->info('🎯 Next Steps:');
        $this->line('1. Deploy to production with updated configuration');
        $this->line('2. Visit: https://mewsai.fumbo.ai/test-onesignal');
        $this->line('3. Allow notifications and get your Player ID');
        $this->line('4. Test notifications from admin panel');
        $this->line('5. Monitor delivery rates and user engagement');
        
        $this->line('');
        $this->info('📚 Documentation: Check ONESIGNAL_PRODUCTION_SETUP.md');
        
        return 0;
    }
}