<?php

namespace App\Console\Commands;

use App\Models\Election\Election;
use App\Models\Election\ElectionResultRegional;
use App\Models\Election\ElectionScrapedData;
use App\Models\Election\Region;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class Sync2020From3News extends Command
{
    protected $signature = 'election:sync-2020-3news {--threshold=0.5 : Max NPP % difference to consider a match (use Option B data if exceeded)} {--dry-run : Show what would change without updating}';
    protected $description = 'Compare seeded 2020 presidential regional data with 3 News scraped data; overwrite with Option B (3 News) where there is a mixup. Run election:import-3news first.';

    public function handle(): int
    {
        $election = Election::where('year', 2020)->where('type', 'presidential')->first();
        if (!$election) {
            $this->error('2020 presidential election not found. Run ElectionDataSeeder first.');
            return self::FAILURE;
        }

        $scraped = ElectionScrapedData::where('election_id', $election->id)
            ->where('data_type', '3news_presidential_2020')
            ->first();
        if (!$scraped || empty($scraped->payload['regional_npp_pct_2020'])) {
            $this->error('No 3 News 2020 data found. Run: npm run scrape:3news then php artisan election:import-3news');
            return self::FAILURE;
        }

        $threshold = (float) $this->option('threshold');
        $dryRun = $this->option('dry-run');
        $regions3 = $scraped->payload['regional_npp_pct_2020'];

        $nppId = DB::table('parties')->where('short_name', 'NPP')->value('id');
        $ndcId = DB::table('parties')->where('short_name', 'NDC')->value('id');
        $othersId = DB::table('parties')->where('short_name', 'OTH')->value('id');
        if (!$nppId || !$ndcId || !$othersId) {
            $this->error('Parties NPP/NDC/OTH not found.');
            return self::FAILURE;
        }

        $regions = Region::all()->keyBy('name');
        $mixups = [];
        $comparison = [];

        foreach ($regions3 as $r3) {
            $regionName = $r3['region'] ?? '';
            $nppPct3 = isset($r3['npp_pct_2020']) ? (float) $r3['npp_pct_2020'] : null;
            if ($regionName === '' || $nppPct3 === null) {
                continue;
            }

            $region = $regions->get($regionName);
            if (!$region) {
                $this->warn("Region not in DB: {$regionName}");
                continue;
            }

            $nppResult = ElectionResultRegional::where('election_id', $election->id)
                ->where('region_id', $region->id)
                ->where('party_id', $nppId)
                ->first();
            $ndcResult = ElectionResultRegional::where('election_id', $election->id)
                ->where('region_id', $region->id)
                ->where('party_id', $ndcId)
                ->first();
            $othersResult = ElectionResultRegional::where('election_id', $election->id)
                ->where('region_id', $region->id)
                ->where('party_id', $othersId)
                ->first();

            $totalInRegion = $nppResult?->total_votes_in_region ?? ($ndcResult?->total_votes_in_region ?? 0);
            $nppPctSeed = $nppResult ? (float) $nppResult->percentage : 0;
            $diff = abs($nppPctSeed - $nppPct3);
            $comparison[] = [
                'region' => $regionName,
                'seeded_npp_pct' => $nppPctSeed,
                '3news_npp_pct' => $nppPct3,
                'diff' => round($diff, 2),
                'mixup' => $diff > $threshold,
            ];

            if ($diff > $threshold) {
                $mixups[] = [
                    'region_id' => $region->id,
                    'region_name' => $regionName,
                    'total_votes' => $totalInRegion,
                    'npp_pct_3news' => $nppPct3,
                    'ndc_pct_3news' => round(100 - $nppPct3, 2),
                ];
            }
        }

        $this->table(
            ['Region', 'Seeded NPP %', '3 News NPP %', 'Diff', 'Mixup?'],
            array_map(fn ($c) => [
                $c['region'],
                $c['seeded_npp_pct'],
                $c['3news_npp_pct'],
                $c['diff'],
                $c['mixup'] ? 'Yes' : 'No',
            ], $comparison)
        );

        if (empty($mixups)) {
            $this->info('No mixups found (all within threshold ' . $threshold . '%). Seeded data matches 3 News. Nothing to update.');
            return self::SUCCESS;
        }

        $this->info('Mixups found: ' . count($mixups) . ' region(s). Using Option B (3 News) data for these.');
        if ($dryRun) {
            $this->warn('Dry run: no changes written. Run without --dry-run to apply.');
            return self::SUCCESS;
        }

        DB::transaction(function () use ($election, $mixups, $nppId, $ndcId, $othersId) {
            foreach ($mixups as $m) {
                $total = (int) $m['total_votes'];
                if ($total <= 0) {
                    continue;
                }
                $nppPct = (float) $m['npp_pct_3news'];
                $ndcPct = (float) $m['ndc_pct_3news'];
                $nppVotes = (int) round($total * $nppPct / 100);
                $ndcVotes = (int) round($total * $ndcPct / 100);
                $othersVotes = max(0, $total - $nppVotes - $ndcVotes);
                $othersPct = $total > 0 ? round(100 - $nppPct - $ndcPct, 2) : 0;

                ElectionResultRegional::where('election_id', $election->id)
                    ->where('region_id', $m['region_id'])
                    ->where('party_id', $nppId)
                    ->update([
                        'votes' => $nppVotes,
                        'percentage' => $nppPct,
                        'updated_at' => now(),
                    ]);
                ElectionResultRegional::where('election_id', $election->id)
                    ->where('region_id', $m['region_id'])
                    ->where('party_id', $ndcId)
                    ->update([
                        'votes' => $ndcVotes,
                        'percentage' => $ndcPct,
                        'updated_at' => now(),
                    ]);
                $othersRow = ElectionResultRegional::where('election_id', $election->id)
                    ->where('region_id', $m['region_id'])
                    ->where('party_id', $othersId)
                    ->first();
                if ($othersRow) {
                    $othersRow->update([
                        'votes' => $othersVotes,
                        'percentage' => $othersPct,
                        'updated_at' => now(),
                    ]);
                }
            }
        });

        $this->info('Updated ' . count($mixups) . ' region(s) with 3 News data.');
        return self::SUCCESS;
    }
}
