<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class AdController extends Controller
{
    /**
     * Display a listing of ads
     */
    public function index()
    {
        $ads = DB::table('ai_ads')->orderBy('created_at', 'desc')->get();
        return view('admin.ads', compact('ads'));
    }

    /**
     * Show the form for creating a new ad
     */
    public function create()
    {
        return view('admin.create-ad');
    }

    /**
     * Store a newly created ad
     */
    public function store(Request $request)
    {
        try {
            \Log::info('AdController::store - Request received', [
                'method' => $request->method(),
                'url' => $request->url(),
                'data' => $request->all()
            ]);
            $validator = Validator::make($request->all(), [
                'title' => 'required|string|max:255',
                'content' => 'required|string',
                'label' => 'required|string|max:100',
                'cta_text' => 'required|string|max:100',
                'cta_url' => 'required|url',
                'media_type' => 'required|in:text,image,video',
                'media_file' => 'nullable|file|mimes:jpg,jpeg,png,gif,webp,mp4,avi,mov|max:10240',
                'media_alt' => 'nullable|string|max:255',
                'display_duration' => 'required|integer|min:0|max:10000',
                'rotation_weight' => 'required|integer|min:1|max:10',
                'is_active' => 'nullable|in:on,1,true',
                'start_date' => 'nullable|date',
                'end_date' => 'nullable|date|after:start_date',
                'max_impressions' => 'nullable|integer|min:1',
            ]);

            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }

            $data = $request->except('media_file');
            $data['is_active'] = $request->has('is_active') ? 1 : 0;

            // Handle media upload
            if ($request->hasFile('media_file')) {
                $file = $request->file('media_file');
                $filename = time() . '_' . $file->getClientOriginalName();
                $file->move(public_path('assets/images/ads'), $filename);
                $data['media_url'] = asset('assets/images/ads/' . $filename);
            }

            // Handle targeting rules
            if ($request->has('targeting_rules')) {
                $data['targeting_rules'] = json_encode($request->targeting_rules);
            }

            // Add timestamps
            $data['created_at'] = now();
            $data['updated_at'] = now();

            $id = DB::table('ai_ads')->insertGetId($data);
            \Log::info('AdController::store - Ad created successfully', ['id' => $id]);

            return redirect()->route('admin.ads')->with('success', 'Ad created successfully!');
        } catch (\Exception $e) {
            \Log::error('AdController::store - Exception occurred', ['message' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return back()->withErrors(['error' => 'Failed to create ad: ' . $e->getMessage()])->withInput();
        }
    }

    /**
     * Show the form for editing an ad
     */
    public function edit($id)
    {
        $ad = DB::table('ai_ads')->find($id);
        
        if (!$ad) {
            return redirect()->route('admin.ads')->with('error', 'Ad not found');
        }

        return view('admin.edit-ad', compact('ad'));
    }

    /**
     * Update the specified ad
     */
    public function update(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'title' => 'required|string|max:255',
                'content' => 'required|string',
                'label' => 'required|string|max:100',
                'cta_text' => 'required|string|max:100',
                'cta_url' => 'required|url',
                'media_type' => 'required|in:text,image,video',
                'media_file' => 'nullable|file|mimes:jpg,jpeg,png,gif,webp,mp4,avi,mov|max:10240',
                'media_alt' => 'nullable|string|max:255',
                'display_duration' => 'required|integer|min:0|max:10000',
                'rotation_weight' => 'required|integer|min:1|max:10',
                'is_active' => 'nullable|in:on,1,true',
                'start_date' => 'nullable|date',
                'end_date' => 'nullable|date|after:start_date',
                'max_impressions' => 'nullable|integer|min:1',
            ]);

            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }

            $data = $request->except('media_file');
            $data['is_active'] = $request->has('is_active') ? 1 : 0;

            // Handle media upload
            if ($request->hasFile('media_file')) {
                // Delete old media file if exists
                $existingAd = DB::table('ai_ads')->where('id', $id)->first();
                if ($existingAd && $existingAd->media_url) {
                    // Extract filename from URL and delete old file
                    $oldFilename = basename($existingAd->media_url);
                    
                    // Try to delete from new assets location
                    $newPath = public_path('assets/images/ads/' . $oldFilename);
                    if (file_exists($newPath)) {
                        unlink($newPath);
                    }
                    
                    // Try to delete from old storage location (for migration compatibility)
                    $oldStoragePath = storage_path('app/public/ads/' . $oldFilename);
                    if (file_exists($oldStoragePath)) {
                        unlink($oldStoragePath);
                    }
                }
                
                $file = $request->file('media_file');
                $filename = time() . '_' . $file->getClientOriginalName();
                $file->move(public_path('assets/images/ads'), $filename);
                $data['media_url'] = asset('assets/images/ads/' . $filename);
            }

            // Handle targeting rules
            if ($request->has('targeting_rules')) {
                $data['targeting_rules'] = json_encode($request->targeting_rules);
            }

            $data['updated_at'] = now();

            DB::table('ai_ads')->where('id', $id)->update($data);

            return redirect()->route('admin.ads')->with('success', 'Ad updated successfully!');
        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Failed to update ad: ' . $e->getMessage()])->withInput();
        }
    }

    /**
     * Remove the specified ad
     */
    public function destroy($id)
    {
        try {
            // Get ad info before deletion
            $ad = DB::table('ai_ads')->where('id', $id)->first();
            
            // Delete associated media file if exists
            if ($ad && $ad->media_url) {
                $oldFilename = basename($ad->media_url);
                
                // Try to delete from new assets location
                $newPath = public_path('assets/images/ads/' . $oldFilename);
                if (file_exists($newPath)) {
                    unlink($newPath);
                }
                
                // Try to delete from old storage location (for migration compatibility)
                $oldStoragePath = storage_path('app/public/ads/' . $oldFilename);
                if (file_exists($oldStoragePath)) {
                    unlink($oldStoragePath);
                }
            }
            
            DB::table('ai_ads')->where('id', $id)->delete();
            return redirect()->route('admin.ads')->with('success', 'Ad deleted successfully!');
        } catch (\Exception $e) {
            return redirect()->route('admin.ads')->with('error', 'Failed to delete ad: ' . $e->getMessage());
        }
    }

    /**
     * Get active ads for frontend
     */
    public function getActiveAds(Request $request)
    {
        $page = $request->get('page', 'chatbot');
        
        $ads = DB::table('ai_ads')
            ->where('is_active', 1)
            ->where(function($query) {
                $query->whereNull('start_date')
                      ->orWhere('start_date', '<=', now());
            })
            ->where(function($query) {
                $query->whereNull('end_date')
                      ->orWhere('end_date', '>=', now());
            })
            ->where(function($query) use ($page) {
                $query->whereNull('targeting_rules')
                      ->orWhere('targeting_rules', 'like', '%"page":"' . $page . '"%');
            })
            ->orderBy('rotation_weight', 'desc')
            ->orderBy('created_at', 'desc')
            ->get();

        if ($ads->isEmpty()) {
            return response()->json(['ad' => null]);
        }

        // Select random ad based on rotation weight
        $totalWeight = $ads->sum('rotation_weight');
        $random = mt_rand(1, $totalWeight);
        $currentWeight = 0;

        foreach ($ads as $ad) {
            $currentWeight += $ad->rotation_weight;
            if ($random <= $currentWeight) {
                // Increment impressions
                DB::table('ai_ads')->where('id', $ad->id)->increment('current_impressions');
                return response()->json(['ad' => $ad]);
            }
        }

        // Fallback to first ad if no ad selected
        $selectedAd = $ads->first();
        if ($selectedAd) {
            DB::table('ai_ads')->where('id', $selectedAd->id)->increment('current_impressions');
        }
        return response()->json(['ad' => $selectedAd]);
    }

    /**
     * Track ad click
     */
    public function trackClick($id)
    {
        try {
            DB::table('ai_ads')->where('id', $id)->increment('clicks');
            return response()->json(['status' => 'success']);
        } catch (\Exception $e) {
            \Log::error('Ad click tracking failed', ['ad_id' => $id, 'error' => $e->getMessage()]);
            return response()->json(['status' => 'error', 'message' => $e->getMessage()], 500);
        }
    }
}