<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class AdminController extends Controller
{
    public function __construct()
    {
        // Simple session-based authentication check
        if (!session()->has('admin_logged_in') && request()->route()->getName() !== 'admin.login') {
            if (request()->route()->getName() !== 'admin.login.post') {
                return redirect()->route('admin.login');
            }
        }
    }

    /**
     * Show admin login page
     */
    public function showLogin()
    {
        if (session()->has('admin_logged_in')) {
            return redirect()->route('admin.dashboard');
        }
        return view('admin.login');
    }

    /**
     * Handle admin login
     */
    public function login(Request $request)
    {
        $request->validate([
            'username' => 'required',
            'password' => 'required'
        ]);

        // Simple hardcoded credentials for now
        // In production, you'd want to use proper user authentication
        if ($request->username === 'admin' && $request->password === 'admin123') {
            session(['admin_logged_in' => true]);
            return redirect()->route('admin.dashboard')->with('success', 'Welcome back!');
        }

        return back()->withErrors(['username' => 'Invalid credentials']);
    }

    /**
     * Handle admin logout
     */
    public function logout()
    {
        session()->forget('admin_logged_in');
        return redirect()->route('admin.login')->with('success', 'Logged out successfully');
    }

    /**
     * Show admin dashboard
     */
    public function dashboard()
    {
        $stats = [
            'total_ads' => DB::table('ai_ads')->count(),
            'active_ads' => DB::table('ai_ads')->where('is_active', true)->count(),
            'total_impressions' => DB::table('ai_ads')->sum('current_impressions'),
            'total_clicks' => DB::table('ai_ads')->sum('clicks'),
        ];

        $recent_ads = DB::table('ai_ads')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        return view('admin.dashboard', compact('stats', 'recent_ads'));
    }

    /**
     * Get default AI settings for fallback values
     */
    private function getDefaultAISettings()
    {
        try {
            $settings = DB::table('ai_settings')
                ->where('category', 'ai')
                ->pluck('setting_value', 'setting_key')
                ->toArray();
            
            return [
                'ai_temperature' => $settings['ai_temperature'] ?? '0.2',
                'ai_max_tokens' => $settings['ai_max_tokens'] ?? '150',
                'ai_model' => $settings['ai_model'] ?? 'gpt-4-turbo',
                'ai_response_speed' => $settings['ai_response_speed'] ?? 'balanced'
            ];
        } catch (\Exception $e) {
            return [
                'ai_temperature' => '0.2',
                'ai_max_tokens' => '150',
                'ai_model' => 'gpt-4-turbo',
                'ai_response_speed' => 'balanced'
            ];
        }
    }

    /**
     * Show settings page
     */
    public function settings()
    {
        $aiSettings = DB::table('ai_settings')->where('category', 'ai')->get();
        $uiSettings = DB::table('ai_settings')->where('category', 'ui')->get();
        $behaviorSettings = DB::table('ai_settings')->where('category', 'behavior')->get();
        
        // Get default values for fallbacks
        $defaultAISettings = $this->getDefaultAISettings();

        return view('admin.settings', compact('aiSettings', 'uiSettings', 'behaviorSettings', 'defaultAISettings'));
    }

    /**
     * Update settings
     */
    public function updateSettings(Request $request)
    {
        // Handle all form fields except CSRF token and method
        $formData = $request->except('_token', '_method');
        
        // Special handling for checkbox fields (they don't send value when unchecked)
        $checkboxFields = ['ai_web_search_enabled'];
        
        foreach ($checkboxFields as $checkboxField) {
            if (!isset($formData[$checkboxField])) {
                // Checkbox is unchecked, set to false
                $formData[$checkboxField] = 'false';
            }
        }
        
        // Update all settings
        foreach ($formData as $key => $value) {
            DB::table('ai_settings')
                ->where('setting_key', $key)
                ->update(['setting_value' => $value]);
        }

        return redirect()->route('admin.settings')->with('success', 'Settings updated successfully!');
    }

    /**
     * Show branding page
     */
    public function branding()
    {
        $colorSettings = DB::table('ai_branding')->where('category', 'colors')->get();
        $imageSettings = DB::table('ai_branding')->where('category', 'images')->get();
        $textSettings = DB::table('ai_branding')->where('category', 'text')->get();
        $uiSettings = DB::table('ai_settings')->where('category', 'ui')->get();

        return view('admin.branding', compact('colorSettings', 'imageSettings', 'textSettings', 'uiSettings'));
    }

    /**
     * Update branding
     */
    public function updateBranding(Request $request)
    {
        foreach ($request->except('_token', '_method') as $key => $value) {
            DB::table('ai_branding')
                ->where('setting_key', $key)
                ->update(['setting_value' => $value]);
        }

        return redirect()->route('admin.branding')->with('success', 'Branding updated successfully!');
    }

    /**
     * Upload branding image
     */
    public function uploadBrandingImage(Request $request)
    {
        $request->validate([
            'image' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:2048'
        ]);

        $file = $request->file('image');
        $filename = time() . '_' . $file->getClientOriginalName();
        $path = $file->storeAs('branding', $filename, 'public');

        return response()->json([
            'success' => true,
            'url' => Storage::url($path)
        ]);
    }

    /**
     * Show reports dashboard
     */
    public function reports(Request $request)
    {
        // Get date filter parameters
        $startDate = $request->get('start_date', now()->subDays(30)->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));
        
        // Validate date range
        if ($startDate > $endDate) {
            $startDate = $endDate;
        }
        
        // Query Report - Top user queries (we'll need to create a query log table)
        $topQueries = $this->getTopQueries($startDate, $endDate);
        
        // View Report - Top read news by views
        $topViewedArticles = $this->getTopViewedArticles($startDate, $endDate);
        
        // Additional reports
        $categoryStats = $this->getCategoryStats($startDate, $endDate);
        $dailyStats = $this->getDailyStats($startDate, $endDate);
        $aiUsageStats = $this->getAIUsageStats($startDate, $endDate);
        
        return view('admin.reports', compact(
            'topQueries', 
            'topViewedArticles', 
            'categoryStats', 
            'dailyStats', 
            'aiUsageStats',
            'startDate',
            'endDate'
        ));
    }

    /**
     * Get top user queries
     */
    private function getTopQueries($startDate = null, $endDate = null)
    {
        try {
            $query = DB::table('query_logs')
                ->select('query', DB::raw('COUNT(*) as count'), DB::raw('MAX(created_at) as last_asked'))
                ->where('was_successful', true)
                ->whereNotNull('query')
                ->where('query', '!=', '');
            
            // Apply date filter if provided
            if ($startDate && $endDate) {
                $query->whereBetween('created_at', [
                    \Carbon\Carbon::parse($startDate)->startOfDay(),
                    \Carbon\Carbon::parse($endDate)->endOfDay()
                ]);
            } else {
                // Default to last 30 days if no dates provided
                $query->where('created_at', '>=', now()->subDays(30));
            }
            
            $results = $query->groupBy('query')
                ->orderBy('count', 'desc')
                ->limit(10)
                ->get();
            
            // If no real data exists, return empty collection
            if ($results->isEmpty()) {
                return collect([]);
            }
            
            return $results->map(function ($item) {
                return [
                    'query' => $item->query,
                    'count' => $item->count,
                    'last_asked' => \Carbon\Carbon::parse($item->last_asked)->format('Y-m-d')
                ];
            });
            
        } catch (\Exception $e) {
            Log::error('Failed to get top queries: ' . $e->getMessage());
            return collect([]);
        }
    }

    /**
     * Get top viewed articles
     */
    private function getTopViewedArticles($startDate = null, $endDate = null)
    {
        try {
            $pdo = new \PDO(
                'mysql:host=' . config('database.connections.mysql.host') . ';dbname=' . config('database.connections.mysql.database') . ';charset=utf8',
                config('database.connections.mysql.username'),
                config('database.connections.mysql.password'),
                [\PDO::ATTR_ERRMODE => \PDO::ERRMODE_EXCEPTION]
            );
            
            $whereClause = "WHERE views > 0";
            $params = [];
            
            if ($startDate && $endDate) {
                $whereClause .= " AND DATE(date_added) BETWEEN :start_date AND :end_date";
                $params['start_date'] = $startDate;
                $params['end_date'] = $endDate;
            }
            
            $stmt = $pdo->prepare("
                SELECT 
                    post_id,
                    post_title,
                    category,
                    sub_category,
                    views,
                    date_added,
                    source
                FROM vw_crawler_mews 
                {$whereClause}
                ORDER BY views DESC 
                LIMIT 20
            ");
            $stmt->execute($params);
            $results = $stmt->fetchAll(\PDO::FETCH_ASSOC);
            
            return collect($results);
        } catch (\Exception $e) {
            Log::error('Failed to get top viewed articles: ' . $e->getMessage());
            return collect([]);
        }
    }

    /**
     * Get category statistics
     */
    private function getCategoryStats($startDate = null, $endDate = null)
    {
        try {
            $pdo = new \PDO(
                'mysql:host=' . config('database.connections.mysql.host') . ';dbname=' . config('database.connections.mysql.database') . ';charset=utf8',
                config('database.connections.mysql.username'),
                config('database.connections.mysql.password'),
                [\PDO::ATTR_ERRMODE => \PDO::ERRMODE_EXCEPTION]
            );
            
            $whereClause = "";
            $params = [];
            
            if ($startDate && $endDate) {
                $whereClause = "WHERE DATE(date_added) BETWEEN :start_date AND :end_date";
                $params['start_date'] = $startDate;
                $params['end_date'] = $endDate;
            }
            
            $stmt = $pdo->prepare("
                SELECT 
                    category,
                    COUNT(*) as article_count,
                    SUM(views) as total_views,
                    AVG(views) as avg_views
                FROM vw_crawler_mews 
                {$whereClause}
                GROUP BY category
                ORDER BY total_views DESC
            ");
            $stmt->execute($params);
            $results = $stmt->fetchAll(\PDO::FETCH_ASSOC);
            
            return collect($results);
        } catch (\Exception $e) {
            Log::error('Failed to get category stats: ' . $e->getMessage());
            return collect([]);
        }
    }

    /**
     * Get daily statistics
     */
    private function getDailyStats($startDate = null, $endDate = null)
    {
        try {
            $pdo = new \PDO(
                'mysql:host=' . config('database.connections.mysql.host') . ';dbname=' . config('database.connections.mysql.database') . ';charset=utf8',
                config('database.connections.mysql.username'),
                config('database.connections.mysql.password'),
                [\PDO::ATTR_ERRMODE => \PDO::ERRMODE_EXCEPTION]
            );
            
            $whereClause = "";
            $params = [];
            
            if ($startDate && $endDate) {
                $whereClause = "WHERE DATE(date_added) BETWEEN :start_date AND :end_date";
                $params['start_date'] = $startDate;
                $params['end_date'] = $endDate;
            } else {
                $whereClause = "WHERE date_added >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)";
            }
            
            $stmt = $pdo->prepare("
                SELECT 
                    DATE(date_added) as date,
                    COUNT(*) as articles_published,
                    SUM(views) as total_views
                FROM vw_crawler_mews 
                {$whereClause}
                GROUP BY DATE(date_added)
                ORDER BY date DESC
                LIMIT 30
            ");
            $stmt->execute($params);
            $results = $stmt->fetchAll(\PDO::FETCH_ASSOC);
            
            return collect($results);
        } catch (\Exception $e) {
            Log::error('Failed to get daily stats: ' . $e->getMessage());
            return collect([]);
        }
    }

    /**
     * Get AI usage statistics
     */
    private function getAIUsageStats($startDate = null, $endDate = null)
    {
        try {
            $dateRange = $startDate && $endDate ? " ({$startDate} to {$endDate})" : " (Last 30 days)";
            
            $query = DB::table('query_logs')
                ->where('was_successful', true);
            
            // Apply date filter if provided
            if ($startDate && $endDate) {
                $query->whereBetween('created_at', [
                    \Carbon\Carbon::parse($startDate)->startOfDay(),
                    \Carbon\Carbon::parse($endDate)->endOfDay()
                ]);
            } else {
                // Default to last 30 days if no dates provided
                $query->where('created_at', '>=', now()->subDays(30));
            }
            
            // Get total queries
            $totalQueries = $query->count();
            
            // Get news queries vs general conversations
            $newsQueries = (clone $query)->where('intent', 'news_query')->count();
            $generalConversations = (clone $query)->where('intent', 'general_conversation')->count();
            
            // Get average response time
            $avgResponseTime = (clone $query)->whereNotNull('response_time_ms')->avg('response_time_ms');
            $avgResponseTimeFormatted = $avgResponseTime ? round($avgResponseTime / 1000, 1) . 's' : 'N/A';
            
            // Get most active hour
            $hourlyStats = (clone $query)
                ->select(DB::raw('HOUR(created_at) as hour'), DB::raw('COUNT(*) as count'))
                ->groupBy('hour')
                ->orderBy('count', 'desc')
                ->first();
            
            $mostActiveHour = $hourlyStats ? 
                sprintf('%02d:00-%02d:00', $hourlyStats->hour, $hourlyStats->hour + 1) : 
                'N/A';
            
            // Get success rate
            $totalAttempts = DB::table('query_logs');
            if ($startDate && $endDate) {
                $totalAttempts->whereBetween('created_at', [
                    \Carbon\Carbon::parse($startDate)->startOfDay(),
                    \Carbon\Carbon::parse($endDate)->endOfDay()
                ]);
            } else {
                $totalAttempts->where('created_at', '>=', now()->subDays(30));
            }
            $totalAttempts = $totalAttempts->count();
            
            $successRate = $totalAttempts > 0 ? 
                round(($totalQueries / $totalAttempts) * 100, 1) . '%' : 
                '100%';
            
            $stats = [
                'total_queries' => $totalQueries,
                'news_queries' => $newsQueries,
                'general_conversations' => $generalConversations,
                'avg_response_time' => $avgResponseTimeFormatted,
                'most_active_hour' => $mostActiveHour,
                'success_rate' => $successRate,
                'date_range' => $dateRange
            ];
            
            return $stats;
        } catch (\Exception $e) {
            Log::error('Failed to get AI usage stats: ' . $e->getMessage());
            return [
                'total_queries' => 0,
                'news_queries' => 0,
                'general_conversations' => 0,
                'avg_response_time' => 'N/A',
                'most_active_hour' => 'N/A',
                'success_rate' => 'N/A',
                'date_range' => $dateRange ?? ' (No data)'
            ];
        }
    }
}