<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\UserNotificationSubscription;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class OneSignalController extends Controller
{
    /**
     * Subscribe user to OneSignal notifications.
     */
    public function subscribe(Request $request)
    {
        $request->validate([
            'player_id' => 'required|string',
            'device_type' => 'nullable|string|in:web,android,ios',
            'browser' => 'nullable|string',
            'timezone' => 'nullable|string',
        ]);

        try {
            $userId = auth()->id();
            $playerId = $request->player_id;

            // Check if subscription already exists
            $subscription = UserNotificationSubscription::where('onesignal_player_id', $playerId)->first();

            if ($subscription) {
                // Update existing subscription
                $subscription->update([
                    'user_id' => $userId,
                    'device_type' => $request->device_type ?? 'web',
                    'browser' => $request->browser,
                    'timezone' => $request->timezone,
                    'is_active' => true,
                    'last_activity_at' => now(),
                ]);
            } else {
                // Create new subscription
                $subscription = UserNotificationSubscription::create([
                    'user_id' => $userId,
                    'onesignal_player_id' => $playerId,
                    'device_type' => $request->device_type ?? 'web',
                    'browser' => $request->browser,
                    'timezone' => $request->timezone,
                    'is_active' => true,
                    'subscribed_at' => now(),
                    'last_activity_at' => now(),
                ]);
            }

            // Set default preferences based on user profile if available
            if ($userId && $subscription->user) {
                $user = $subscription->user;
                
                // Set default categories based on user interests
                if ($user->profile && $user->profile->interests) {
                    $subscription->update([
                        'subscribed_categories' => $user->profile->interests,
                        'subscribed_interests' => $user->profile->interests,
                    ]);
                }
                
                // Set default location based on user profile
                if ($user->profile && $user->profile->location) {
                    $subscription->update([
                        'subscribed_locations' => [$user->profile->location],
                    ]);
                }
            }

            Log::info('OneSignal subscription created/updated', [
                'user_id' => $userId,
                'player_id' => $playerId,
                'subscription_id' => $subscription->id,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Successfully subscribed to notifications',
                'subscription_id' => $subscription->id,
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to create OneSignal subscription', [
                'user_id' => $userId ?? null,
                'player_id' => $request->player_id,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to subscribe to notifications',
            ], 500);
        }
    }

    /**
     * Update notification preferences.
     */
    public function updatePreferences(Request $request)
    {
        $request->validate([
            'player_id' => 'required|string',
            'categories' => 'nullable|array',
            'locations' => 'nullable|array',
            'interests' => 'nullable|array',
            'breaking_news_enabled' => 'nullable|boolean',
            'daily_digest_enabled' => 'nullable|boolean',
            'weekly_summary_enabled' => 'nullable|boolean',
            'marketing_enabled' => 'nullable|boolean',
            'quiet_hours_start' => 'nullable|date_format:H:i',
            'quiet_hours_end' => 'nullable|date_format:H:i',
            'preferred_days' => 'nullable|array',
            'preferred_times' => 'nullable|array',
        ]);

        try {
            $subscription = UserNotificationSubscription::where('onesignal_player_id', $request->player_id)->first();

            if (!$subscription) {
                return response()->json([
                    'success' => false,
                    'message' => 'Subscription not found',
                ], 404);
            }

            $updateData = $request->only([
                'subscribed_categories',
                'subscribed_locations',
                'subscribed_interests',
                'breaking_news_enabled',
                'daily_digest_enabled',
                'weekly_summary_enabled',
                'marketing_enabled',
                'quiet_hours_start',
                'quiet_hours_end',
                'preferred_days',
                'preferred_times',
            ]);

            // Map request fields to database fields
            if ($request->has('categories')) {
                $updateData['subscribed_categories'] = $request->categories;
            }
            if ($request->has('locations')) {
                $updateData['subscribed_locations'] = $request->locations;
            }
            if ($request->has('interests')) {
                $updateData['subscribed_interests'] = $request->interests;
            }

            $subscription->update($updateData);

            Log::info('OneSignal preferences updated', [
                'subscription_id' => $subscription->id,
                'player_id' => $request->player_id,
                'preferences' => $updateData,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Preferences updated successfully',
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to update OneSignal preferences', [
                'player_id' => $request->player_id,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to update preferences',
            ], 500);
        }
    }

    /**
     * Unsubscribe user from notifications.
     */
    public function unsubscribe(Request $request)
    {
        $request->validate([
            'player_id' => 'required|string',
        ]);

        try {
            $subscription = UserNotificationSubscription::where('onesignal_player_id', $request->player_id)->first();

            if ($subscription) {
                $subscription->update(['is_active' => false]);
                
                Log::info('OneSignal subscription deactivated', [
                    'subscription_id' => $subscription->id,
                    'player_id' => $request->player_id,
                ]);
            }

            return response()->json([
                'success' => true,
                'message' => 'Successfully unsubscribed from notifications',
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to unsubscribe from OneSignal', [
                'player_id' => $request->player_id,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to unsubscribe',
            ], 500);
        }
    }

    /**
     * Get notification preferences.
     */
    public function getPreferences(Request $request)
    {
        $request->validate([
            'player_id' => 'required|string',
        ]);

        try {
            $subscription = UserNotificationSubscription::where('onesignal_player_id', $request->player_id)->first();

            if (!$subscription) {
                return response()->json([
                    'success' => false,
                    'message' => 'Subscription not found',
                ], 404);
            }

            return response()->json([
                'success' => true,
                'preferences' => [
                    'categories' => $subscription->subscribed_categories ?? [],
                    'locations' => $subscription->subscribed_locations ?? [],
                    'interests' => $subscription->subscribed_interests ?? [],
                    'breaking_news_enabled' => $subscription->breaking_news_enabled,
                    'daily_digest_enabled' => $subscription->daily_digest_enabled,
                    'weekly_summary_enabled' => $subscription->weekly_summary_enabled,
                    'marketing_enabled' => $subscription->marketing_enabled,
                    'quiet_hours_start' => $subscription->quiet_hours_start,
                    'quiet_hours_end' => $subscription->quiet_hours_end,
                    'preferred_days' => $subscription->preferred_days ?? [],
                    'preferred_times' => $subscription->preferred_times ?? [],
                ],
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to get OneSignal preferences', [
                'player_id' => $request->player_id,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to get preferences',
            ], 500);
        }
    }
}