<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\UserProfile;
use App\Models\UserPreference;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\Password;

class RegisterController extends Controller
{
    /**
     * Show the registration form.
     */
    public function showRegistrationForm()
    {
        return view('auth.register');
    }

    /**
     * Handle user registration.
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'confirmed', Password::defaults()],
            'first_name' => ['nullable', 'string', 'max:255'],
            'last_name' => ['nullable', 'string', 'max:255'],
            'phone' => ['nullable', 'string', 'max:20'],
            'location' => ['nullable', 'string', 'max:255'],
            'interests' => ['nullable', 'string'], // Changed to string since it comes as JSON
            'language_preference' => ['nullable', 'string', 'in:en,twi,ga,ewe'],
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Parse interests from JSON string
        $interests = [];
        if ($request->interests) {
            $interests = json_decode($request->interests, true) ?? [];
        }

        // Generate name from first_name and last_name, or use email
        $name = trim(($request->first_name ?? '') . ' ' . ($request->last_name ?? ''));
        if (empty($name)) {
            $name = explode('@', $request->email)[0]; // Use email username as fallback
        }

        // Create the user
        $user = User::create([
            'name' => $name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
        ]);

        // Create user profile
        $profile = UserProfile::create([
            'user_id' => $user->id,
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'phone' => $request->phone,
            'location' => $request->location,
            'interests' => $interests,
            'language_preference' => $request->language_preference ?? 'en',
            'last_active_at' => now(),
        ]);

        // Create default user preferences
        UserPreference::create([
            'user_id' => $user->id,
            ...UserPreference::getDefaults(),
        ]);

        // Log the user in
        Auth::login($user);

        return redirect()->route('news.chatbot')
            ->with('success', 'Welcome to Mews AI! Your account has been created successfully.');
    }
}