<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\DB;
use PDO;
use App\Services\OpenAIWebService;

class DeepDiveController extends Controller
{
    private $openaiApiKey;
    private $openaiModel;
    private $aiSettings;
    private $openAIWebService;

    public function __construct()
    {
        $this->openaiApiKey = config('services.openai.api_key');
        
        // Get AI settings from database, fallback to config if not found
        $this->aiSettings = $this->getAISettings();
        $this->openaiModel = $this->aiSettings['ai_model'];
        
        // Initialize web search service
        $this->openAIWebService = app(OpenAIWebService::class);
        
        // Debug log to verify the model being used
        Log::info('DeepDive initialized with model: ' . $this->openaiModel);
    }

    /**
     * Get default AI settings from database
     */
    private function getDefaultAISettings()
    {
        try {
            $settings = DB::table('ai_settings')
                ->where('category', 'ai')
                ->pluck('setting_value', 'setting_key')
                ->toArray();
            
            return [
                'ai_temperature' => (float) ($settings['ai_temperature'] ?? 0.2),
                'ai_max_tokens' => (int) ($settings['ai_max_tokens'] ?? 150),
                'ai_model' => $settings['ai_model'] ?? config('services.openai.model', 'gpt-4-turbo'),
                'ai_response_speed' => $settings['ai_response_speed'] ?? 'balanced'
            ];
        } catch (\Exception $e) {
            Log::error('Failed to load default AI settings from database: ' . $e->getMessage());
            // If database fails, try to get from config or use minimal defaults
            return [
                'ai_temperature' => (float) config('services.openai.temperature', 0.2),
                'ai_max_tokens' => (int) config('services.openai.max_tokens', 150),
                'ai_model' => config('services.openai.model', 'gpt-4-turbo'),
                'ai_response_speed' => config('services.openai.response_speed', 'balanced')
            ];
        }
    }

    /**
     * Get AI settings from database
     */
    private function getAISettings()
    {
        try {
            $settings = DB::table('ai_settings')
                ->where('category', 'ai')
                ->pluck('setting_value', 'setting_key')
                ->toArray();
            
            // Get defaults from database
            $defaults = $this->getDefaultAISettings();
            
            return [
                'ai_temperature' => (float) ($settings['ai_temperature'] ?? $defaults['ai_temperature']),
                'ai_max_tokens' => (int) ($settings['ai_max_tokens'] ?? $defaults['ai_max_tokens']),
                'ai_model' => $settings['ai_model'] ?? $defaults['ai_model'],
                'ai_response_speed' => $settings['ai_response_speed'] ?? $defaults['ai_response_speed']
            ];
        } catch (\Exception $e) {
            Log::error('Failed to load AI settings from database: ' . $e->getMessage());
            return $this->getDefaultAISettings();
        }
    }

    /**
     * Get UI settings from database
     */
    private function getUISettings()
    {
        try {
            $settings = DB::table('ai_settings')
                ->where('category', 'ui')
                ->pluck('setting_value', 'setting_key')
                ->toArray();
            
            return [
                'typing_speed' => $settings['typing_speed'] ?? 'medium',
                'ad_display_delay' => $settings['ad_display_delay'] ?? 'short',
                'auto_scroll' => $settings['auto_scroll'] ?? 'enabled',
                'sound_notifications' => $settings['sound_notifications'] ?? 'disabled'
            ];
        } catch (\Exception $e) {
            Log::error('Failed to load UI settings from database: ' . $e->getMessage());
            return [
                'typing_speed' => 'medium',
                'ad_display_delay' => 'short',
                'auto_scroll' => 'enabled',
                'sound_notifications' => 'disabled'
            ];
        }
    }

    public function show($articleId)
    {
        try {
            Log::info('Deep Dive show called with article ID: ' . $articleId);
            
            // Fetch the specific article from database
            $article = $this->fetchArticle($articleId);
            
            if (!$article) {
                Log::error('Article not found for ID: ' . $articleId);
                return redirect()->route('news.chatbot')->with('error', 'Article not found');
            }

            Log::info('Article found: ' . $article['title']);
            
            // Increment view counter for this article
            $this->incrementViewCounter($articleId);
            
            // Track reading history for authenticated users
            $this->trackReadingHistory($articleId, $article);
            
            // Get UI settings for typing speed and other UI preferences
            $uiSettings = $this->getUISettings();
            
            return view('deep-dive', compact('article', 'uiSettings'));
            
        } catch (\Exception $e) {
            Log::error('Deep Dive show error: ' . $e->getMessage());
            return redirect()->route('news.chatbot')->with('error', 'Error loading article');
        }
    }

    /**
     * Get article data for API (mobile app)
     */
    public function getArticle($articleId)
    {
        try {
            Log::info('Deep Dive API getArticle called with article ID: ' . $articleId);
            
            // Fetch the specific article from database
            $article = $this->fetchArticle($articleId);
            
            if (!$article) {
                Log::error('Article not found for ID: ' . $articleId);
                return response()->json([
                    'success' => false,
                    'error' => 'Article not found'
                ], 404);
            }

            Log::info('Article found via API: ' . $article['title']);
            
            // Increment view counter for this article
            $this->incrementViewCounter($articleId);
            
            // Track reading history for authenticated users
            $this->trackReadingHistory($articleId, $article);
            
            return response()->json([
                'success' => true,
                'data' => $article
            ]);
            
        } catch (\Exception $e) {
            Log::error('Deep Dive API getArticle error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Error loading article: ' . $e->getMessage()
            ], 500);
        }
    }

    public function chat(Request $request, $articleId)
    {
        try {
            $userQuery = $request->input('message');
            $sessionId = 'deep_dive_' . $articleId . '_' . session()->getId();
            
            // Debug log to verify the model being used
            Log::info('Processing deep dive chat request with model: ' . $this->openaiModel);
            
            // Fetch the article for context
            $article = $this->fetchArticle($articleId);
            if (!$article) {
                return response()->json([
                    'success' => false,
                    'error' => 'Article not found'
                ]);
            }

            // Generate AI response with article context
            $response = $this->generateDeepDiveResponse($userQuery, $article, $sessionId);
            
            return response()->json([
                'success' => true,
                'response' => $response,
                'article_id' => $articleId
            ]);

        } catch (\Exception $e) {
            Log::error('Deep Dive chat error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'error' => 'An error occurred while processing your request'
            ]);
        }
    }

    private function fetchArticle($articleId)
    {
        try {
            Log::info('Fetching article from database with ID: ' . $articleId);
            
            $pdo = new PDO(
                'mysql:host=' . config('database.connections.mysql.host') . ';dbname=' . config('database.connections.mysql.database') . ';charset=utf8',
                config('database.connections.mysql.username'),
                config('database.connections.mysql.password'),
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
            
            $stmt = $pdo->prepare("
                SELECT * FROM vw_crawler_mews 
                WHERE post_id = :article_id
                LIMIT 1
            ");
            $stmt->execute(['article_id' => $articleId]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($result) {
                Log::info('Database result found: ' . json_encode($result));
                
                // Clean the content to remove unnecessary HTML elements
                $cleanContent = $this->cleanArticleContent($result['post_content'] ?? '');
                
                return [
                    'id' => $result['post_id'] ?? 0,
                    'title' => $result['post_title'] ?? 'Untitled Article',
                    'category' => $result['category'] ?? 'General',
                    'sub_category' => $result['sub_category'] ?? '',
                    'content' => $cleanContent,
                    'date' => $this->formatDate($result['date_added'] ?? time()),
                    'source' => $result['post_source'] ?? '#',
                    'image' => $result['image'] ?? '',
                    'views' => $result['views'] ?? 0
                ];
            }
            
            Log::error('No database result found for article ID: ' . $articleId);
            return null;
            
        } catch (\Exception $e) {
            Log::error('Database error in fetchArticle: ' . $e->getMessage());
            return null;
        }
    }
    
    private function cleanArticleContent($content)
    {
        // First, try to extract just the main article content by looking for paragraphs
        preg_match_all('/<p[^>]*>(.*?)<\/p>/si', $content, $matches);
        
        if (!empty($matches[1])) {
            // Filter out paragraphs that are likely navigation, ads, or unwanted content
            $filteredParagraphs = [];
            foreach ($matches[1] as $paragraph) {
                $cleanParagraphText = trim(strip_tags($paragraph)); // Only strip for length/keyword check
                
                // Skip paragraphs that are too short (likely navigation/ads)
                if (strlen($cleanParagraphText) < 20) {
                    continue;
                }
                
                // Skip paragraphs that contain common unwanted text
                $unwantedTexts = [
                    'Share This Post',
                    'Follow Us',
                    'Sign up',
                    'You May Also Like',
                    'Up Next',
                    'Related',
                    'Leave a comment',
                    'Post a comment',
                    'Add your comment',
                    'TAGS:',
                    'Subscribe',
                    'Newsletter',
                    'Advertisement',
                    'Sponsored'
                ];
                
                $skipParagraph = false;
                foreach ($unwantedTexts as $unwanted) {
                    if (stripos($cleanParagraphText, $unwanted) !== false) {
                        $skipParagraph = true;
                        break;
                    }
                }
                
                if (!$skipParagraph) {
                    $filteredParagraphs[] = '<p>' . $paragraph . '</p>'; // Keep original HTML of paragraph
                }
            }
            
            if (!empty($filteredParagraphs)) {
                $content = implode('', $filteredParagraphs);
            }
        }
        
        // If we still have content, do minimal cleaning while preserving media
        if (!empty($content)) {
            // Remove only very specific unwanted elements
            $unwantedPatterns = [
                // Remove SVG icons
                '/<svg[^>]*>.*?<\/svg>/si',
                // Remove empty divs
                '/<div[^>]*>\s*<\/div>/',
                // Remove script tags
                '/<script[^>]*>.*?<\/script>/si',
                // Remove style tags
                '/<style[^>]*>.*?<\/style>/si',
                // Remove "Read also" sections with links to other articles
                '/<div[^>]*>.*?read\s+also.*?<\/div>/si',
                '/<p[^>]*>.*?read\s+also.*?<\/p>/si',
                '/<section[^>]*>.*?read\s+also.*?<\/section>/si',
                // Remove related articles sections
                '/<div[^>]*>.*?related\s+articles.*?<\/div>/si',
                '/<p[^>]*>.*?related\s+articles.*?<\/p>/si',
                '/<section[^>]*>.*?related\s+articles.*?<\/section>/si',
                // Remove "You may also like" sections
                '/<div[^>]*>.*?you\s+may\s+also\s+like.*?<\/div>/si',
                '/<p[^>]*>.*?you\s+may\s+also\s+like.*?<\/p>/si',
                '/<section[^>]*>.*?you\s+may\s+also\s+like.*?<\/section>/si',
                // Remove "READ MORE:" sections
                '/<div[^>]*>.*?read\s+more.*?<\/div>/si',
                '/<p[^>]*>.*?read\s+more.*?<\/p>/si',
                '/<section[^>]*>.*?read\s+more.*?<\/section>/si',
                '/<a[^>]*>.*?read\s+more.*?<\/a>/si',
            ];
            
            foreach ($unwantedPatterns as $pattern) {
                $content = preg_replace($pattern, '', $content);
            }
            
            // Convert "Play Video" links to nice buttons
            $content = preg_replace_callback(
                '/<a[^>]*>.*?play\s+video.*?<\/a>/si',
                function($matches) {
                    return '<button class="play-video-btn" onclick="playVideo(this)" style="background: linear-gradient(135deg, #0a57d4, #2d72f0); color: white; border: none; padding: 12px 24px; border-radius: 25px; cursor: pointer; font-weight: 500; margin: 1rem 0; display: inline-flex; align-items: center; gap: 8px; box-shadow: 0 4px 15px rgba(10, 87, 212, 0.3); transition: all 0.3s ease;"><i class="feather" data-feather="play" style="width: 16px; height: 16px;"></i> Play Video</button>';
                },
                $content
            );
            
            // Clean up excessive whitespace
            $content = preg_replace('/\s+/', ' ', $content);
            $content = preg_replace('/\n\s*\n/', "\n", $content);
            
            // Clean up any remaining empty tags
            $content = preg_replace('/<[^\/>][^>]*><\/[^>]*>/', '', $content);
        }
        
        // Final cleanup
        $content = trim($content);
        
        // If content is empty after cleaning, return a fallback message
        if (empty(strip_tags($content))) {
            return '<p>Article content is being processed. Please check back later.</p>';
        }
        
        return $content;
    }

    private function generateDeepDiveResponse($userQuery, $article, $sessionId)
    {
        try {
            // Prepare article context for AI
            $articleContext = "
Article Title: {$article['title']}
Category: {$article['category']}
Sub-Category: {$article['sub_category']}
Date: {$article['date']}
Content: {$article['content']}
            ";

            // Use web search service for current information
            if ($this->isWebSearchEnabled() && $this->openAIWebService->isWebSearchEnabled()) {
                $messages = [
                    [
                        'role' => 'system',
                        'content' => 'You are an expert news analyst. You have access to a specific news article and can search the web for current information. 

RESPONSE STRATEGY:
1. **FIRST**: Answer using information from the provided article
2. **THEN**: Use web search to find current, relevant information to supplement your answer
3. **BE CONCISE**: Keep responses brief and to the point
4. **NO SOURCES**: Do not include source links or citations

RESPONSE FORMAT:
- Start with article information (if relevant)
- Add current information from web search
- Keep responses under 200 words
- Focus on essential facts only
- Be direct and informative'
                    ],
                    [
                        'role' => 'user',
                        'content' => "Article Context:\n{$articleContext}\n\nUser Question: {$userQuery}\n\nPlease answer using the article information first, then search for current information to provide a complete, concise response."
                    ]
                ];

                $response = $this->openAIWebService->generateResponseWithWebSearch($messages, [
                    'max_tokens' => 200,
                    'temperature' => 0.3
                ]);

                if ($response && !empty($response['content'])) {
                    return $response['content'];
                }
            }

            // Fallback to regular model if web search fails
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->openaiApiKey,
                'Content-Type' => 'application/json',
            ])->post('https://api.openai.com/v1/chat/completions', [
                'model' => 'gpt-4o-mini', // Use regular model for fallback
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are an expert news analyst. Answer questions about the provided article concisely. Keep responses under 200 words and focus on essential information only.'
                    ],
                    [
                        'role' => 'user',
                        'content' => "Article:\n{$articleContext}\n\nQuestion: {$userQuery}"
                    ]
                ],
                'max_tokens' => 200,
                'temperature' => 0.3
            ]);

            if ($response->successful()) {
                $data = $response->json();
                return $data['choices'][0]['message']['content'] ?? '';
            }

        } catch (\Exception $e) {
            Log::error('OpenAI deep dive response generation error: ' . $e->getMessage());
        }

        return "I'm having trouble analyzing this article right now. Please try again or ask a different question.";
    }

    /**
     * Check if web search is enabled in settings
     */
    private function isWebSearchEnabled()
    {
        try {
            $webSearchEnabled = DB::table('ai_settings')
                ->where('setting_key', 'ai_web_search_enabled')
                ->value('setting_value');
            
            return $webSearchEnabled === 'true';
        } catch (\Exception $e) {
            Log::error('Failed to check web search setting: ' . $e->getMessage());
            return true; // Default to enabled if setting not found
        }
    }

    private function formatDate($timestamp)
    {
        if (is_numeric($timestamp)) {
            return date('F j, Y', $timestamp);
        }
        return date('F j, Y', strtotime($timestamp));
    }

    /**
     * Increment view counter for an article
     */
    private function incrementViewCounter($articleId)
    {
        try {
            $pdo = new PDO(
                'mysql:host=' . config('database.connections.mysql.host') . ';dbname=' . config('database.connections.mysql.database') . ';charset=utf8',
                config('database.connections.mysql.username'),
                config('database.connections.mysql.password'),
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
            
            // First, try to update in in_posts table (for articles from feeds)
            $stmt = $pdo->prepare("UPDATE in_posts SET post_hits = post_hits + 1 WHERE post_id = :article_id");
            $stmt->execute(['article_id' => $articleId]);
            
            // If no rows were affected, try to update in mews_posts table (for local articles)
            if ($stmt->rowCount() === 0) {
                $stmt = $pdo->prepare("UPDATE mews_posts SET post_hits = post_hits + 1 WHERE id = :article_id");
                $stmt->execute(['article_id' => $articleId]);
            }
            
            Log::info('View counter incremented for article ID: ' . $articleId . ' (rows affected: ' . $stmt->rowCount() . ')');
            
        } catch (\Exception $e) {
            Log::error('Failed to increment view counter for article ID ' . $articleId . ': ' . $e->getMessage());
        }
    }

    /**
     * Track reading history for authenticated users
     */
    private function trackReadingHistory($articleId, $article)
    {
        try {
            // Only track for authenticated users
            if (!auth()->check()) {
                return;
            }

            $user = auth()->user();
            
            // Check if this article is already in the user's reading history
            $existingHistory = \App\Models\ReadingHistory::where('user_id', $user->id)
                ->where('article_id', $articleId)
                ->first();

            if ($existingHistory) {
                // Update the existing record with new read time
                $existingHistory->update([
                    'read_at' => now(),
                    'access_method' => 'deep_dive'
                ]);
            } else {
                // Create new reading history record
                \App\Models\ReadingHistory::create([
                    'user_id' => $user->id,
                    'article_id' => $articleId,
                    'article_title' => $article['title'],
                    'article_category' => $article['category'] ?? null,
                    'article_source' => $article['source'] ?? null,
                    'read_at' => now(),
                    'access_method' => 'deep_dive'
                ]);
            }

            Log::info('Reading history tracked for user ' . $user->id . ' and article ' . $articleId);
            
        } catch (\Exception $e) {
            Log::error('Failed to track reading history for article ID ' . $articleId . ': ' . $e->getMessage());
        }
    }
}
