<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Models\Election\Election;
use App\Models\Election\Party;
use App\Models\Election\Region;
use App\Models\Election\Candidate;
use App\Models\Election\Poll;
use App\Models\Election\PollResult;
use App\Models\Election\AIPrediction;
use App\Models\Election\ElectionResultRegional;
use App\Models\Election\ParliamentarySeat;
use App\Models\Election\SocialMention;
use App\Models\Election\ElectionChatLog;
use App\Models\Election\ElectionScrapedData;
use App\Models\Election\RegionCensus;
use App\Services\ElectionAIService;

class ElectionController extends Controller
{
    private $aiService;

    public function __construct(ElectionAIService $aiService)
    {
        $this->aiService = $aiService;
    }

    /**
     * Election Intelligence Dashboard
     */
    public function dashboard(Request $request)
    {
        // Get selected election: by id or by year+type (Peace FM: presidential + presidential_runoff only)
        $electionList = Election::whereIn('type', ['presidential', 'presidential_runoff'])
            ->orderByDesc('year')
            ->get();

        if ($request->query('election_id')) {
            $election = $electionList->firstWhere('id', (int) $request->query('election_id'));
        } elseif ($request->query('year') && $request->query('type')) {
            $election = $electionList->firstWhere('year', (int) $request->query('year'))
                ?: Election::where('year', (int) $request->query('year'))
                    ->where('type', $request->query('type'))
                    ->first();
        } else {
            $election = $electionList->first();
        }
        $election = $election ?? $electionList->first();

        $selectedYear = $election?->year;
        $selectedType = $election?->type;
        $availableElections = $electionList->groupBy('year');

        // Get parties
        $parties = Party::whereIn('short_name', ['NPP', 'NDC'])->get()->keyBy('short_name');

        // Race overview and regional breakdown: Peace FM scraped data only
        $raceOverview = $this->getRaceOverviewFromScraped($election);
        $regionalBreakdown = $this->getRegionalBreakdownFromScraped($election);

        // Get recent polls
        $recentPolls = Poll::where('election_id', $election?->id)
            ->with(['results.party', 'results.candidate'])
            ->orderByDesc('date_conducted')
            ->limit(5)
            ->get();

        // Get AI predictions
        $predictions = AIPrediction::where('election_id', $election?->id)
            ->with(['party', 'candidate'])
            ->orderByDesc('win_probability')
            ->get();

        // Get social sentiment summary
        $socialSentiment = $this->getSocialSentimentSummary($election);

        // Parliamentary seats (if applicable)
        $parliamentarySeats = null;
        if ($selectedType === 'parliamentary' && $election) {
            $parliamentarySeats = ParliamentarySeat::where('election_id', $election->id)
                ->with('party')
                ->orderByDesc('seats_won')
                ->get();
        }

        // Census: 2000 for 2000–2012, 2010 for 2016/2020, 2021 for 2024; fall back to next available year if not in DB
        $preferredYear = $this->censusYearForElection($election?->year);
        $result = $this->getCensusDataWithFallback($preferredYear);
        $censusData = $result['data'];
        $censusYear = $result['year'];
        $totalPopulation = $censusData->sum('population');
        $censusSource = $this->censusSourceLabel($censusYear);
        if ($censusYear !== $preferredYear && $totalPopulation > 0) {
            $censusSource .= sprintf(' (%d data not in database)', $preferredYear);
        }

        // Merge census population into regional breakdown by region name
        $popByRegion = $censusData->keyBy('name');
        foreach ($regionalBreakdown as &$r) {
            $r['population'] = $popByRegion->get($r['region'])?->population;
        }
        unset($r);

        // Peace FM scraped data for selected election (single source)
        $scraped = $election ? [
            'national' => ElectionScrapedData::where('election_id', $election->id)->where('data_type', 'national_summary')->first()?->payload,
            'swing' => ElectionScrapedData::where('election_id', $election->id)->where('data_type', 'swing_constituencies')->first()?->payload,
            'marginal' => ElectionScrapedData::where('election_id', $election->id)->where('data_type', 'marginal_wins')->first()?->payload,
            'regional_analysis' => ElectionScrapedData::where('election_id', $election->id)->where('data_type', 'regional_analysis')->first()?->payload,
            'region_turnout' => ElectionScrapedData::where('election_id', $election->id)->where('data_type', 'region_turnout')->first()?->payload,
            'rejected_ballots' => ElectionScrapedData::where('election_id', $election->id)->where('data_type', 'rejected_ballots')->first()?->payload,
        ] : null;

        // Comparison years for regional table: use exact years from scraped data when present (e.g. 2016, 2020)
        $comparisonPrevYear = $election ? $election->year - 4 : null;
        $comparisonCurrYear = $election ? (string) $election->year : null;
        if ($election && $election->type === 'presidential_runoff') {
            $comparisonPrevYear = $election->year;
            $comparisonCurrYear = $election->year . ' R/o';
        }
        $regAnalysis = $scraped['regional_analysis']['regional_comparison'] ?? [];
        if ($election && $scraped && count($regAnalysis) > 0) {
            $first = $regAnalysis[0];
            if (isset($first['npp_2016'], $first['npp_2020'])) {
                $comparisonPrevYear = '2016';
                $comparisonCurrYear = '2020';
            } elseif (isset($first['npp_2012'], $first['npp_2016'])) {
                $comparisonPrevYear = '2012';
                $comparisonCurrYear = '2016';
            } elseif (isset($first['npp_1'], $first['npp_2'])) {
                $comparisonPrevYear = $election->year - 4;
                $comparisonCurrYear = $election->year;
            }
        }

        // Historical trend from Peace FM scraped national data only
        $historicalData = $this->getHistoricalComparisonFromScraped();

        return view('elections.dashboard', compact(
            'election',
            'electionList',
            'availableElections',
            'parties',
            'raceOverview',
            'regionalBreakdown',
            'recentPolls',
            'predictions',
            'socialSentiment',
            'historicalData',
            'parliamentarySeats',
            'selectedYear',
            'selectedType',
            'censusData',
            'totalPopulation',
            'censusYear',
            'censusSource',
            'scraped',
            'comparisonPrevYear',
            'comparisonCurrYear'
        ));
    }

    /**
     * Census year to use for an election: 2000 for 2000–2012, 2010 for 2016/2020, 2021 for 2024.
     */
    private function censusYearForElection(?int $year): int
    {
        if (! $year) {
            return 2021;
        }
        if ($year <= 2012) {
            return 2000;
        }
        if ($year <= 2020) {
            return 2010;
        }
        return 2021;
    }

    /**
     * Census data for preferred year, falling back to 2010 then 2021 when not in DB.
     * Returns ['data' => collection, 'year' => actual census year used].
     */
    private function getCensusDataWithFallback(int $preferredYear): array
    {
        $yearsToTry = array_unique(array_merge(
            [$preferredYear],
            [2010, 2021]
        ));
        foreach ($yearsToTry as $year) {
            $data = $this->getCensusDataForYear($year);
            if ($data->isNotEmpty()) {
                return ['data' => $data, 'year' => $year];
            }
        }
        return ['data' => collect(), 'year' => $preferredYear];
    }

    /**
     * Census data (regions with population) for a given census year.
     * Uses region_census when available; falls back to regions.population for 2021.
     */
    private function getCensusDataForYear(int $censusYear): \Illuminate\Support\Collection
    {
        $rows = RegionCensus::where('census_year', $censusYear)
            ->with('region')
            ->get();

        if ($rows->isNotEmpty()) {
            return $rows->map(function ($rc) {
                return (object) [
                    'name' => $rc->region?->name ?? 'Unknown',
                    'population' => (int) $rc->population,
                ];
            })->sortByDesc('population')->values();
        }

        if ($censusYear === 2021) {
            return Region::whereNotNull('population')->where('population', '>', 0)
                ->orderByDesc('population')
                ->get()
                ->map(fn ($r) => (object) ['name' => $r->name, 'population' => (int) $r->population]);
        }

        return collect();
    }

    /**
     * Human-readable source label for census year.
     */
    private function censusSourceLabel(int $censusYear): string
    {
        return sprintf('Ghana Statistical Service, %d PHC', $censusYear);
    }

    /**
     * Race overview from Peace FM scraped national_summary only.
     */
    private function getRaceOverviewFromScraped(?Election $election): array
    {
        if (! $election) {
            return ['npp' => 0, 'ndc' => 0, 'others' => 0, 'type' => 'no_data'];
        }
        $row = ElectionScrapedData::where('election_id', $election->id)->where('data_type', 'national_summary')->first();
        $nat = $row?->payload['national'] ?? [];
        $nppV = (int) ($nat['npp']['votes'] ?? 0);
        $ndcV = (int) ($nat['ndc']['votes'] ?? 0);
        if (($nppV + $ndcV) <= 0) {
            return ['npp' => 0, 'ndc' => 0, 'others' => 0, 'type' => 'no_data'];
        }
        // Use Peace FM percentages when present (matches source exactly); else derive from votes
        $nppPct = isset($nat['npp']['percentage']) ? round((float) $nat['npp']['percentage'], 2) : round(($nppV / ($nppV + $ndcV)) * 100, 2);
        $ndcPct = isset($nat['ndc']['percentage']) ? round((float) $nat['ndc']['percentage'], 2) : round(($ndcV / ($nppV + $ndcV)) * 100, 2);
        $others = max(0, round(100 - $nppPct - $ndcPct, 2));
        return [
            'npp' => $nppPct,
            'ndc' => $ndcPct,
            'others' => $others,
            'npp_votes' => $nppV,
            'ndc_votes' => $ndcV,
            'total_votes' => $nppV + $ndcV,
            'type' => 'actual',
            'winner' => $nppV >= $ndcV ? 'NPP' : 'NDC',
        ];
    }

    /**
     * Regional breakdown from Peace FM scraped regional_analysis or national.regional_votes.
     */
    private function getRegionalBreakdownFromScraped(?Election $election): array
    {
        if (! $election) {
            return [];
        }
        $regionalAnalysis = ElectionScrapedData::where('election_id', $election->id)->where('data_type', 'regional_analysis')->first()?->payload;
        $regComp = $regionalAnalysis['regional_comparison'] ?? [];
        if (count($regComp) > 0) {
            $breakdown = [];
            foreach ($regComp as $r) {
                $nppPct = (float) ($r['npp_2'] ?? $r['npp_2020'] ?? $r['npp_curr'] ?? 0);
                $ndcPct = (float) ($r['ndc_2'] ?? $r['ndc_2020'] ?? $r['ndc_curr'] ?? 0);
                if ($nppPct <= 0 && $ndcPct <= 0) {
                    $nppPct = (float) ($r['npp_1'] ?? $r['npp_2016'] ?? 50);
                    $ndcPct = (float) ($r['ndc_1'] ?? $r['ndc_2016'] ?? 50);
                }
                $breakdown[] = [
                    'region' => $r['region'] ?? '',
                    'region_code' => '',
                    'npp_pct' => $nppPct,
                    'ndc_pct' => $ndcPct,
                    'npp_votes' => 0,
                    'ndc_votes' => 0,
                    'total_votes_in_region' => 0,
                    'winner' => $nppPct >= $ndcPct ? 'NPP' : 'NDC',
                    'population' => null,
                ];
            }
            usort($breakdown, fn ($a, $b) => $b['npp_pct'] <=> $a['npp_pct']);
            return $breakdown;
        }
        $nationalRow = ElectionScrapedData::where('election_id', $election->id)->where('data_type', 'national_summary')->first()?->payload;
        $regionalVotes = $nationalRow['regional_votes'] ?? [];
        if (count($regionalVotes) > 0) {
            $breakdown = [];
            foreach ($regionalVotes as $r) {
                $nppPct = (float) ($r['npp_pct'] ?? 0);
                $ndcPct = (float) ($r['ndc_pct'] ?? 0);
                $breakdown[] = [
                    'region' => $r['region'] ?? '',
                    'region_code' => '',
                    'npp_pct' => $nppPct,
                    'ndc_pct' => $ndcPct,
                    'npp_votes' => (int) ($r['npp_votes'] ?? 0),
                    'ndc_votes' => (int) ($r['ndc_votes'] ?? 0),
                    'total_votes_in_region' => (int) (($r['npp_votes'] ?? 0) + ($r['ndc_votes'] ?? 0)),
                    'winner' => $nppPct >= $ndcPct ? 'NPP' : 'NDC',
                    'population' => null,
                ];
            }
            usort($breakdown, fn ($a, $b) => $b['npp_pct'] <=> $a['npp_pct']);
            return $breakdown;
        }
        return [];
    }

    /**
     * Historical comparison from Peace FM scraped national_summary only.
     */
    private function getHistoricalComparisonFromScraped(): array
    {
        $elections = Election::whereIn('type', ['presidential', 'presidential_runoff'])
            ->where('status', 'completed')
            ->orderBy('year')
            ->get();
        $comparison = [];
        foreach ($elections as $election) {
            $row = ElectionScrapedData::where('election_id', $election->id)->where('data_type', 'national_summary')->first();
            $nat = $row?->payload['national'] ?? [];
            $nppV = (int) ($nat['npp']['votes'] ?? 0);
            $ndcV = (int) ($nat['ndc']['votes'] ?? 0);
            $total = $nppV + $ndcV;
            if ($total <= 0) {
                continue;
            }
            $nppPct = round(($nppV / $total) * 100, 2);
            $ndcPct = round(($ndcV / $total) * 100, 2);
            $comparison[] = [
                'year' => $election->year . ($election->type === 'presidential_runoff' ? ' R/o' : ''),
                'name' => $election->name,
                'npp_pct' => $nppPct,
                'ndc_pct' => $ndcPct,
                'winner' => $nppV >= $ndcV ? 'NPP' : 'NDC',
            ];
        }
        return $comparison;
    }

    /**
     * Get race overview data
     */
    private function getRaceOverview(?Election $election, $parties): array
    {
        if (!$election) {
            return ['npp' => 0, 'ndc' => 0, 'others' => 0, 'type' => 'no_data'];
        }

        // For completed elections, use actual results
        if ($election->status === 'completed') {
            $results = ElectionResultRegional::where('election_id', $election->id)
                ->selectRaw('party_id, SUM(votes) as total_votes')
                ->groupBy('party_id')
                ->get();

            $totalVotes = $results->sum('total_votes');
            
            $nppId = $parties['NPP']?->id;
            $ndcId = $parties['NDC']?->id;
            
            $nppVotes = $results->where('party_id', $nppId)->first()?->total_votes ?? 0;
            $ndcVotes = $results->where('party_id', $ndcId)->first()?->total_votes ?? 0;
            $othersVotes = $totalVotes - $nppVotes - $ndcVotes;

            return [
                'npp' => $totalVotes > 0 ? round(($nppVotes / $totalVotes) * 100, 2) : 0,
                'ndc' => $totalVotes > 0 ? round(($ndcVotes / $totalVotes) * 100, 2) : 0,
                'others' => $totalVotes > 0 ? round(($othersVotes / $totalVotes) * 100, 2) : 0,
                'npp_votes' => $nppVotes,
                'ndc_votes' => $ndcVotes,
                'total_votes' => $totalVotes,
                'type' => 'actual',
                'winner' => $nppVotes > $ndcVotes ? 'NPP' : 'NDC',
            ];
        }

        // For upcoming elections, use latest poll or prediction
        $latestPoll = Poll::where('election_id', $election->id)
            ->with('results.party')
            ->orderByDesc('date_conducted')
            ->first();

        if ($latestPoll) {
            $nppResult = $latestPoll->results->first(fn($r) => $r->party?->short_name === 'NPP');
            $ndcResult = $latestPoll->results->first(fn($r) => $r->party?->short_name === 'NDC');

            return [
                'npp' => $nppResult?->percentage ?? 0,
                'ndc' => $ndcResult?->percentage ?? 0,
                'others' => 100 - (($nppResult?->percentage ?? 0) + ($ndcResult?->percentage ?? 0)),
                'type' => 'poll',
                'poll_source' => $latestPoll->pollster_name,
                'poll_date' => $latestPoll->date_conducted,
            ];
        }

        return ['npp' => 0, 'ndc' => 0, 'others' => 0, 'type' => 'no_data'];
    }

    /**
     * Get regional breakdown
     */
    private function getRegionalBreakdown(?Election $election): array
    {
        if (!$election) {
            return [];
        }

        $regions = Region::all();
        $breakdown = [];

        foreach ($regions as $region) {
            $results = ElectionResultRegional::where('election_id', $election->id)
                ->where('region_id', $region->id)
                ->with('party')
                ->get();

            $nppResult = $results->first(fn($r) => $r->party?->short_name === 'NPP');
            $ndcResult = $results->first(fn($r) => $r->party?->short_name === 'NDC');

            $breakdown[] = [
                'region' => $region->name,
                'region_code' => $region->code,
                'npp_pct' => $nppResult?->percentage ?? 0,
                'ndc_pct' => $ndcResult?->percentage ?? 0,
                'npp_votes' => $nppResult?->votes ?? 0,
                'ndc_votes' => $ndcResult?->votes ?? 0,
                'total_votes' => $nppResult?->total_votes_in_region ?? 0,
                'winner' => ($nppResult?->is_region_winner ?? false) ? 'NPP' : (($ndcResult?->is_region_winner ?? false) ? 'NDC' : '-'),
                'population' => $region->population, // 2021 census
            ];
        }

        // Sort by NPP percentage descending (strongholds first)
        usort($breakdown, fn($a, $b) => $b['npp_pct'] <=> $a['npp_pct']);

        return $breakdown;
    }

    /**
     * Get social sentiment summary
     */
    private function getSocialSentimentSummary(?Election $election): array
    {
        if (!$election) {
            return ['available' => false];
        }

        $mentions = SocialMention::where('election_id', $election->id)
            ->where('captured_at', '>=', now()->subDays(7))
            ->get();

        if ($mentions->isEmpty()) {
            return ['available' => false, 'message' => 'Social sentiment data coming soon'];
        }

        $parties = Party::whereIn('short_name', ['NPP', 'NDC'])->get();
        $partySentiment = [];

        foreach ($parties as $party) {
            $partyMentions = $mentions->where('party_id', $party->id);
            $total = $partyMentions->count();
            
            $partySentiment[$party->short_name] = [
                'total' => $total,
                'positive' => $partyMentions->where('sentiment', 'positive')->count(),
                'negative' => $partyMentions->where('sentiment', 'negative')->count(),
                'neutral' => $partyMentions->where('sentiment', 'neutral')->count(),
                'positive_pct' => $total > 0 ? round(($partyMentions->where('sentiment', 'positive')->count() / $total) * 100) : 0,
            ];
        }

        return [
            'available' => true,
            'total_mentions' => $mentions->count(),
            'party_sentiment' => $partySentiment,
            'trending_hashtags' => $mentions->pluck('hashtags')->filter()->flatten()->countBy()->sortDesc()->take(5),
        ];
    }

    /**
     * Get historical comparison data
     */
    private function getHistoricalComparison(string $type): array
    {
        $elections = Election::where('type', $type)
            ->where('status', 'completed')
            ->orderByDesc('year')
            ->limit(6)
            ->get();

        $parties = Party::whereIn('short_name', ['NPP', 'NDC'])->get()->keyBy('short_name');
        $comparison = [];

        foreach ($elections as $election) {
            $results = ElectionResultRegional::where('election_id', $election->id)
                ->selectRaw('party_id, SUM(votes) as total_votes')
                ->groupBy('party_id')
                ->get();

            $totalVotes = $results->sum('total_votes');
            $nppVotes = $results->where('party_id', $parties['NPP']?->id)->first()?->total_votes ?? 0;
            $ndcVotes = $results->where('party_id', $parties['NDC']?->id)->first()?->total_votes ?? 0;

            $comparison[] = [
                'year' => $election->year,
                'name' => $election->name,
                'npp_pct' => $totalVotes > 0 ? round(($nppVotes / $totalVotes) * 100, 2) : 0,
                'ndc_pct' => $totalVotes > 0 ? round(($ndcVotes / $totalVotes) * 100, 2) : 0,
                'winner' => $nppVotes > $ndcVotes ? 'NPP' : 'NDC',
            ];
        }

        return array_reverse($comparison); // Chronological order
    }

    /**
     * Regional detail view
     */
    public function region(Request $request, string $regionCode)
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $region = Region::where('code', $regionCode)->firstOrFail();
        
        $selectedYear = $request->get('year');
        $election = $selectedYear 
            ? Election::where('year', $selectedYear)->presidential()->first()
            : Election::presidential()->completed()->orderByDesc('year')->first();

        // Get all results for this region across elections
        $historicalResults = ElectionResultRegional::where('region_id', $region->id)
            ->with(['election', 'party', 'candidate'])
            ->whereHas('election', fn($q) => $q->presidential())
            ->get()
            ->groupBy('election_id');

        $availableElections = Election::presidential()->orderByDesc('year')->get();

        return view('elections.region', compact(
            'region',
            'election',
            'historicalResults',
            'availableElections'
        ));
    }

    /**
     * Compare two elections
     */
    public function compare(Request $request)
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $year1 = $request->get('year1', 2020);
        $year2 = $request->get('year2', 2016);
        $type = $request->get('type', 'presidential');

        $election1 = Election::where('year', $year1)->where('type', $type)->first();
        $election2 = Election::where('year', $year2)->where('type', $type)->first();

        $comparison = [];
        $regions = Region::all();

        foreach ($regions as $region) {
            $results1 = ElectionResultRegional::where('election_id', $election1?->id)
                ->where('region_id', $region->id)
                ->with('party')
                ->get();
            
            $results2 = ElectionResultRegional::where('election_id', $election2?->id)
                ->where('region_id', $region->id)
                ->with('party')
                ->get();

            $npp1 = $results1->first(fn($r) => $r->party?->short_name === 'NPP');
            $ndc1 = $results1->first(fn($r) => $r->party?->short_name === 'NDC');
            $npp2 = $results2->first(fn($r) => $r->party?->short_name === 'NPP');
            $ndc2 = $results2->first(fn($r) => $r->party?->short_name === 'NDC');

            $comparison[] = [
                'region' => $region->name,
                'year1' => [
                    'npp' => $npp1?->percentage ?? 0,
                    'ndc' => $ndc1?->percentage ?? 0,
                    'winner' => ($npp1?->is_region_winner ?? false) ? 'NPP' : 'NDC',
                ],
                'year2' => [
                    'npp' => $npp2?->percentage ?? 0,
                    'ndc' => $ndc2?->percentage ?? 0,
                    'winner' => ($npp2?->is_region_winner ?? false) ? 'NPP' : 'NDC',
                ],
                'swing' => [
                    'npp' => ($npp1?->percentage ?? 0) - ($npp2?->percentage ?? 0),
                    'ndc' => ($ndc1?->percentage ?? 0) - ($ndc2?->percentage ?? 0),
                ],
            ];
        }

        $availableYears = Election::where('type', $type)
            ->distinct()
            ->pluck('year')
            ->sort()
            ->values();

        return view('elections.compare', compact(
            'election1',
            'election2',
            'comparison',
            'availableYears',
            'year1',
            'year2',
            'type'
        ));
    }

    /**
     * Polls listing and management
     */
    public function polls(Request $request)
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $electionId = $request->get('election_id');
        
        $query = Poll::with(['results.party', 'results.candidate', 'uploadedBy'])
            ->orderByDesc('date_conducted');
        
        if ($electionId) {
            $query->where('election_id', $electionId);
        }

        $polls = $query->paginate(20);
        $elections = Election::orderByDesc('year')->get();

        return view('elections.polls', compact('polls', 'elections', 'electionId'));
    }

    /**
     * Add new poll
     */
    public function storePoll(Request $request)
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $request->validate([
            'election_id' => 'required|exists:elections,id',
            'pollster_name' => 'required|string|max:255',
            'pollster_organization' => 'nullable|string|max:255',
            'methodology' => 'nullable|string',
            'sample_size' => 'nullable|integer|min:1',
            'margin_of_error' => 'nullable|numeric|min:0|max:100',
            'date_conducted' => 'required|date',
            'date_published' => 'nullable|date',
            'source_url' => 'nullable|url',
            'disclaimer' => 'nullable|string',
            'results' => 'required|array',
            'results.*.party_id' => 'required|exists:parties,id',
            'results.*.percentage' => 'required|numeric|min:0|max:100',
        ]);

        $poll = Poll::create([
            'election_id' => $request->election_id,
            'pollster_name' => $request->pollster_name,
            'pollster_organization' => $request->pollster_organization,
            'methodology' => $request->methodology,
            'sample_size' => $request->sample_size,
            'margin_of_error' => $request->margin_of_error,
            'date_conducted' => $request->date_conducted,
            'date_published' => $request->date_published,
            'source_url' => $request->source_url,
            'disclaimer' => $request->disclaimer ?? 'Poll results are estimates and subject to margin of error. Actual election outcomes may differ.',
            'uploaded_by' => Auth::id(),
            'is_verified' => false,
        ]);

        foreach ($request->results as $result) {
            PollResult::create([
                'poll_id' => $poll->id,
                'party_id' => $result['party_id'],
                'percentage' => $result['percentage'],
            ]);
        }

        return redirect()->route('elections.polls')
            ->with('success', 'Poll added successfully. It will be reviewed for verification.');
    }

    /**
     * Election AI Chat
     */
    public function chat(Request $request)
    {
        if (!Auth::check()) {
            return response()->json(['error' => 'Unauthorized'], 401);
        }

        $request->validate([
            'query' => 'required|string|max:1000',
            'election_id' => 'nullable|exists:elections,id',
        ]);

        $startTime = microtime(true);
        $query = $request->input('query');
        $electionId = $request->input('election_id');
        $sessionId = $request->input('session_id', 'election_' . Auth::id());

        try {
            $result = $this->aiService->answerElectionQuestion($query, $electionId);
            $responseTime = round((microtime(true) - $startTime) * 1000);

            // Log the chat
            ElectionChatLog::create([
                'user_id' => Auth::id(),
                'session_id' => $sessionId,
                'query' => $query,
                'response' => $result['response'] ?? null,
                'intent' => $this->detectIntent($query),
                'data_sources_used' => $result['sources_used'] ?? [],
                'response_time_ms' => $responseTime,
                'was_successful' => $result['success'],
                'error_message' => $result['error'] ?? null,
                'user_ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ]);

            return response()->json($result);

        } catch (\Exception $e) {
            Log::error('Election chat error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'error' => 'An error occurred while processing your question'
            ], 500);
        }
    }

    /**
     * Generate AI prediction
     */
    public function generatePrediction(Request $request)
    {
        if (!Auth::check()) {
            return response()->json(['error' => 'Unauthorized'], 401);
        }

        $request->validate([
            'election_id' => 'required|exists:elections,id',
        ]);

        $election = Election::findOrFail($request->election_id);
        $result = $this->aiService->generatePrediction($election);

        return response()->json($result);
    }

    /**
     * Detect intent from query
     */
    private function detectIntent(string $query): string
    {
        $query = strtolower($query);

        if (str_contains($query, 'predict') || str_contains($query, 'will win') || str_contains($query, 'chance')) {
            return 'prediction';
        }
        if (str_contains($query, 'compare') || str_contains($query, 'vs') || str_contains($query, 'versus')) {
            return 'comparison';
        }
        if (str_contains($query, 'poll') || str_contains($query, 'survey')) {
            return 'poll_check';
        }
        if (str_contains($query, 'history') || str_contains($query, 'past') || preg_match('/\b(19|20)\d{2}\b/', $query)) {
            return 'historical';
        }
        if (str_contains($query, 'region') || str_contains($query, 'constituency') || str_contains($query, 'ashanti') || str_contains($query, 'volta')) {
            return 'regional';
        }

        return 'general';
    }

    /**
     * API: Get election data for charts
     */
    public function apiGetElectionData(Request $request, int $electionId)
    {
        $election = Election::with([
            'regionalResults.region',
            'regionalResults.party',
            'predictions.party',
        ])->findOrFail($electionId);

        return response()->json([
            'election' => $election,
            'regional_data' => $this->getRegionalBreakdownFromScraped($election),
        ]);
    }
}
