<?php

namespace App\Models\Election;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Election extends Model
{
    protected $fillable = [
        'name',
        'type',
        'year',
        'election_date',
        'status',
        'total_registered_voters',
        'total_votes_cast',
        'voter_turnout',
        'total_valid_votes',
        'total_rejected_votes',
        'notes',
    ];

    protected $casts = [
        'election_date' => 'date',
        'year' => 'integer',
        'total_registered_voters' => 'integer',
        'total_votes_cast' => 'integer',
        'voter_turnout' => 'decimal:2',
        'total_valid_votes' => 'integer',
        'total_rejected_votes' => 'integer',
    ];

    public function candidates(): HasMany
    {
        return $this->hasMany(ElectionCandidate::class);
    }

    public function nationalResults(): HasMany
    {
        return $this->hasMany(ElectionResultNational::class);
    }

    public function regionalResults(): HasMany
    {
        return $this->hasMany(ElectionResultRegional::class);
    }

    public function constituencyResults(): HasMany
    {
        return $this->hasMany(ElectionResultConstituency::class);
    }

    public function parliamentarySeats(): HasMany
    {
        return $this->hasMany(ParliamentarySeat::class);
    }

    public function polls(): HasMany
    {
        return $this->hasMany(Poll::class);
    }

    public function predictions(): HasMany
    {
        return $this->hasMany(AIPrediction::class);
    }

    public function socialMentions(): HasMany
    {
        return $this->hasMany(SocialMention::class);
    }

    /**
     * Scope for presidential elections
     */
    public function scopePresidential($query)
    {
        return $query->whereIn('type', ['presidential', 'presidential_runoff']);
    }

    /**
     * Scope for parliamentary elections
     */
    public function scopeParliamentary($query)
    {
        return $query->where('type', 'parliamentary');
    }

    /**
     * Scope for completed elections
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Scope for upcoming elections
     */
    public function scopeUpcoming($query)
    {
        return $query->where('status', 'upcoming');
    }

    /**
     * Get the winner of this election
     */
    public function getWinner()
    {
        return $this->nationalResults()
            ->where('is_winner', true)
            ->with('candidate', 'party')
            ->first();
    }

    /**
     * Get results grouped by party
     */
    public function getResultsByParty()
    {
        return $this->nationalResults()
            ->with('party')
            ->orderByDesc('votes')
            ->get();
    }

    /**
     * Get regional breakdown
     */
    public function getRegionalBreakdown()
    {
        return $this->regionalResults()
            ->with(['region', 'party'])
            ->get()
            ->groupBy('region_id');
    }

    /**
     * Check if this is a runoff election
     */
    public function getIsRunoffAttribute(): bool
    {
        return str_contains($this->type, 'runoff');
    }

    /**
     * Get display name for the election type
     */
    public function getTypeDisplayAttribute(): string
    {
        return match($this->type) {
            'presidential' => 'Presidential',
            'presidential_runoff' => 'Presidential Runoff',
            'parliamentary' => 'Parliamentary',
            default => ucfirst($this->type),
        };
    }
}
