<?php

namespace App\Models\Election;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Models\User;

class Poll extends Model
{
    protected $fillable = [
        'election_id',
        'pollster_name',
        'pollster_organization',
        'methodology',
        'sample_size',
        'margin_of_error',
        'date_conducted',
        'date_published',
        'source_url',
        'source_document',
        'is_verified',
        'notes',
        'disclaimer',
        'uploaded_by',
    ];

    protected $casts = [
        'date_conducted' => 'date',
        'date_published' => 'date',
        'sample_size' => 'integer',
        'margin_of_error' => 'decimal:2',
        'is_verified' => 'boolean',
    ];

    public function election(): BelongsTo
    {
        return $this->belongsTo(Election::class);
    }

    public function uploadedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'uploaded_by');
    }

    public function results(): HasMany
    {
        return $this->hasMany(PollResult::class);
    }

    /**
     * Get results ordered by percentage
     */
    public function getResultsRanked()
    {
        return $this->results()
            ->with(['candidate', 'party'])
            ->orderByDesc('percentage')
            ->get();
    }

    /**
     * Get the leading candidate/party
     */
    public function getLeader()
    {
        return $this->results()
            ->with(['candidate', 'party'])
            ->orderByDesc('percentage')
            ->first();
    }

    /**
     * Get formatted methodology summary
     */
    public function getMethodologySummaryAttribute(): string
    {
        $parts = [];
        
        if ($this->sample_size) {
            $parts[] = "n=" . number_format($this->sample_size);
        }
        
        if ($this->margin_of_error) {
            $parts[] = "±{$this->margin_of_error}%";
        }
        
        if ($this->methodology) {
            $parts[] = $this->methodology;
        }
        
        return implode(' · ', $parts);
    }

    /**
     * Get days until election (if upcoming)
     */
    public function getDaysUntilElectionAttribute(): ?int
    {
        if (!$this->election || !$this->election->election_date) {
            return null;
        }
        
        return now()->diffInDays($this->election->election_date, false);
    }
}
