<?php

namespace App\Models\Election;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class SocialMention extends Model
{
    protected $fillable = [
        'election_id',
        'candidate_id',
        'party_id',
        'platform',
        'content_snippet',
        'post_url',
        'post_id',
        'sentiment',
        'sentiment_score',
        'engagement_score',
        'likes',
        'retweets',
        'replies',
        'hashtags',
        'author_handle',
        'author_followers',
        'posted_at',
        'captured_at',
    ];

    protected $casts = [
        'sentiment_score' => 'decimal:3',
        'engagement_score' => 'integer',
        'likes' => 'integer',
        'retweets' => 'integer',
        'replies' => 'integer',
        'author_followers' => 'integer',
        'posted_at' => 'datetime',
        'captured_at' => 'datetime',
    ];

    public function election(): BelongsTo
    {
        return $this->belongsTo(Election::class);
    }

    public function candidate(): BelongsTo
    {
        return $this->belongsTo(Candidate::class);
    }

    public function party(): BelongsTo
    {
        return $this->belongsTo(Party::class);
    }

    /**
     * Scope for positive sentiment
     */
    public function scopePositive($query)
    {
        return $query->where('sentiment', 'positive');
    }

    /**
     * Scope for negative sentiment
     */
    public function scopeNegative($query)
    {
        return $query->where('sentiment', 'negative');
    }

    /**
     * Scope for neutral sentiment
     */
    public function scopeNeutral($query)
    {
        return $query->where('sentiment', 'neutral');
    }

    /**
     * Scope for Twitter/X platform
     */
    public function scopeTwitter($query)
    {
        return $query->where('platform', 'twitter');
    }

    /**
     * Get sentiment color for display
     */
    public function getSentimentColorAttribute(): string
    {
        return match($this->sentiment) {
            'positive' => '#22c55e',
            'negative' => '#ef4444',
            'neutral' => '#6b7280',
            default => '#6b7280',
        };
    }

    /**
     * Get total engagement
     */
    public function getTotalEngagementAttribute(): int
    {
        return ($this->likes ?? 0) + ($this->retweets ?? 0) + ($this->replies ?? 0);
    }
}
