<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ReadingHistory extends Model
{
    use HasFactory;

    protected $table = 'reading_history';

    protected $fillable = [
        'user_id',
        'article_id',
        'article_title',
        'article_category',
        'article_source',
        'access_method',
        'reading_time_seconds',
        'completed',
        'scroll_percentage',
        'read_at',
    ];

    protected $casts = [
        'completed' => 'boolean',
        'read_at' => 'datetime',
    ];

    /**
     * Get the user that owns the reading history.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get reading time in a human-readable format.
     */
    public function getReadingTimeFormattedAttribute()
    {
        if (!$this->reading_time_seconds) {
            return 'Unknown';
        }

        $minutes = floor($this->reading_time_seconds / 60);
        $seconds = $this->reading_time_seconds % 60;

        if ($minutes > 0) {
            return $minutes . 'm ' . $seconds . 's';
        }

        return $seconds . 's';
    }

    /**
     * Get the reading engagement level.
     */
    public function getEngagementLevelAttribute()
    {
        if ($this->completed && $this->scroll_percentage >= 90) {
            return 'high';
        } elseif ($this->scroll_percentage >= 50) {
            return 'medium';
        } else {
            return 'low';
        }
    }

    /**
     * Scope for recent reading history.
     */
    public function scopeRecent($query, $days = 7)
    {
        return $query->where('read_at', '>=', now()->subDays($days));
    }

    /**
     * Scope for completed articles.
     */
    public function scopeCompleted($query)
    {
        return $query->where('completed', true);
    }

    /**
     * Scope for high engagement articles.
     */
    public function scopeHighEngagement($query)
    {
        return $query->where('completed', true)
                    ->where('scroll_percentage', '>=', 90);
    }
}