<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class UserPreference extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'theme',
        'font_size',
        'auto_play_videos',
        'show_images',
        'news_display_mode',
        'articles_per_page',
        'category_preferences',
        'blocked_sources',
        'blocked_keywords',
        'ai_suggestions',
        'trending_alerts',
        'timezone',
    ];

    protected $casts = [
        'auto_play_videos' => 'boolean',
        'show_images' => 'boolean',
        'category_preferences' => 'array',
        'blocked_sources' => 'array',
        'blocked_keywords' => 'array',
        'ai_suggestions' => 'boolean',
        'trending_alerts' => 'boolean',
    ];

    /**
     * Get the user that owns the preferences.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get default preferences for a new user.
     */
    public static function getDefaults()
    {
        return [
            'theme' => 'light',
            'font_size' => 'medium',
            'auto_play_videos' => false,
            'show_images' => true,
            'news_display_mode' => 'cards',
            'articles_per_page' => 10,
            'category_preferences' => [],
            'blocked_sources' => [],
            'blocked_keywords' => [],
            'ai_suggestions' => true,
            'trending_alerts' => true,
            'timezone' => 'Africa/Accra',
        ];
    }

    /**
     * Check if a category is preferred.
     */
    public function prefersCategory($category)
    {
        $preferences = $this->category_preferences ?? [];
        return in_array($category, $preferences);
    }

    /**
     * Check if a source is blocked.
     */
    public function isSourceBlocked($source)
    {
        $blocked = $this->blocked_sources ?? [];
        return in_array($source, $blocked);
    }

    /**
     * Check if content contains blocked keywords.
     */
    public function containsBlockedKeywords($content)
    {
        $blocked = $this->blocked_keywords ?? [];
        
        foreach ($blocked as $keyword) {
            if (stripos($content, $keyword) !== false) {
                return true;
            }
        }
        
        return false;
    }
}