<?php

namespace App\Services;

use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\URL;

class ImageService
{
    /**
     * Get the URL for a stored image
     */
    public static function getImageUrl($path, $disk = 'public', $cacheBust = true)
    {
        if (!$path) {
            return null;
        }

        // If it's already a full URL, return as is
        if (filter_var($path, FILTER_VALIDATE_URL)) {
            return $path;
        }

        // Use Storage to generate the URL
        $url = Storage::disk($disk)->url($path);
        
        // Add cache-busting parameter if requested
        if ($cacheBust && file_exists(storage_path('app/public/' . $path))) {
            $timestamp = filemtime(storage_path('app/public/' . $path));
            $url .= '?v=' . $timestamp;
        }
        
        return $url;
    }

    /**
     * Get the URL for a stored image with fallback
     */
    public static function getImageUrlWithFallback($path, $fallback = null, $disk = 'public')
    {
        $url = self::getImageUrl($path, $disk);
        
        if ($url && self::imageExists($path, $disk)) {
            return $url;
        }

        return $fallback;
    }

    /**
     * Check if an image exists
     */
    public static function imageExists($path, $disk = 'public')
    {
        if (!$path) {
            return false;
        }

        return Storage::disk($disk)->exists($path);
    }

    /**
     * Store an uploaded file and return the path
     */
    public static function storeUploadedFile($file, $directory = 'uploads', $disk = 'public')
    {
        if (!$file) {
            return null;
        }

        $filename = time() . '_' . $file->getClientOriginalName();
        $path = $file->storeAs($directory, $filename, $disk);
        
        return $path;
    }

    /**
     * Delete an image file
     */
    public static function deleteImage($path, $disk = 'public')
    {
        if (!$path) {
            return false;
        }

        if (Storage::disk($disk)->exists($path)) {
            return Storage::disk($disk)->delete($path);
        }

        return false;
    }

    /**
     * Get a default avatar URL based on initials
     */
    public static function getDefaultAvatarUrl($name, $size = 128, $background = 'ff6b35', $color = 'fff')
    {
        $initials = '';
        $nameParts = explode(' ', trim($name));
        
        foreach ($nameParts as $part) {
            if (!empty($part)) {
                $initials .= strtoupper(substr($part, 0, 1));
            }
        }
        
        if (empty($initials)) {
            $initials = 'U';
        }

        // Add cache-busting parameter to prevent browser caching issues
        $timestamp = time();
        return "https://ui-avatars.com/api/?name={$initials}&background={$background}&color={$color}&size={$size}&v={$timestamp}";
    }
}
