<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class OpenAIWebService
{
    private $apiKey;
    private $model;
    private $webSearchEnabled;

    public function __construct()
    {
        $this->apiKey = config('services.openai.api_key');
        
        // Get AI settings from database, fallback to config if not found
        $aiSettings = $this->getAISettings();
        $this->model = $aiSettings['ai_model'] ?? config('services.openai.model');
        
        $this->webSearchEnabled = config('services.openai.web_search', true);
    }
    
    /**
     * Get AI settings from database
     */
    private function getAISettings()
    {
        try {
            $settings = \DB::table('ai_settings')->pluck('setting_value', 'setting_key')->toArray();
            return $settings;
        } catch (\Exception $e) {
            \Log::error('Failed to get AI settings from database: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Generate response with web search capability
     */
    public function generateResponseWithWebSearch(array $messages, array $options = [])
    {
        try {
            $payload = [
                'model' => $this->model,
                'messages' => $messages,
                'max_completion_tokens' => $options['max_tokens'] ?? 1000,
            ];

            // Only add temperature for models that support it
            if (!str_contains($this->model, 'search')) {
                $payload['temperature'] = $options['temperature'] ?? 0.7;
            }

            // For GPT-5 mini, web search is built-in and doesn't require special tools
            // The model automatically searches the web when it needs current information

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->post('https://api.openai.com/v1/chat/completions', $payload);

            if ($response->successful()) {
                $data = $response->json();
                return $this->processResponse($data);
            }

            Log::error('OpenAI API request failed', [
                'status' => $response->status(),
                'response' => $response->body(),
                'model' => $this->model,
            ]);

            return null;

        } catch (\Exception $e) {
            Log::error('OpenAI API error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Process the response and clean up source references
     */
    private function processResponse($data)
    {
        $response = [
            'content' => '',
            'web_search_results' => [],
            'sources' => []
        ];

        if (isset($data['choices'][0]['message']['content'])) {
            $content = $data['choices'][0]['message']['content'];
            
            // Remove source references and links
            $content = $this->removeSourceReferences($content);
            
            $response['content'] = $content;
        }

        return $response;
    }
    
    /**
     * Remove source references and links from content
     */
    private function removeSourceReferences($content)
    {
        // Remove markdown links like [text](url)
        $content = preg_replace('/\[([^\]]+)\]\([^)]+\)/', '$1', $content);
        
        // Remove URLs that start with http/https
        $content = preg_replace('/https?:\/\/[^\s]+/', '', $content);
        
        // Remove source references in parentheses like (graphic.com.gh)
        $content = preg_replace('/\([^)]*\.(com|org|net|gov|gh|uk|us)[^)]*\)/', '', $content);
        
        // Remove "Sources:" sections
        $content = preg_replace('/\*\*Sources:\*\*.*$/s', '', $content);
        $content = preg_replace('/## Recent Developments.*$/s', '', $content);
        
        // Remove standalone domain references
        $content = preg_replace('/\b[a-zA-Z0-9-]+\.(com|org|net|gov|gh|uk|us)\b/', '', $content);
        
        // Clean up extra whitespace and punctuation
        $content = preg_replace('/\s+/', ' ', $content);
        $content = preg_replace('/\s*\.\s*\.\s*/', '. ', $content);
        $content = preg_replace('/\s*,\s*,/', ',', $content);
        $content = trim($content);
        
        return $content;
    }

    /**
     * Generate news response with real-time web search
     */
    public function generateNewsResponse($query, $context = [])
    {
        $messages = [
            [
                'role' => 'system',
                'content' => 'You are a professional Ghana-focused news assistant. Use web search to find the most current information about the user\'s query. 

GHANA DEFAULT CONTEXT:
- This is a Ghana-focused news platform, so when users ask general questions without specifying a country, assume they mean Ghana
- Examples: "Who won the last election?" → "Who won the last Ghana election?", "What\'s the inflation rate?" → "What\'s Ghana\'s inflation rate?"
- Only use this default when NO country is explicitly mentioned
- If a user mentions another country (Nigeria, USA, etc.), respect that context

Provide accurate, up-to-date information in a concise format. Keep responses brief and to the point. Do not include source links, references, or citations. Focus on delivering the essential information clearly and directly.'
            ],
            [
                'role' => 'user',
                'content' => $query
            ]
        ];

        return $this->generateResponseWithWebSearch($messages, [
            'max_tokens' => 300,
            'temperature' => 0.3
        ]);
    }

    /**
     * Check if web search is available
     */
    public function isWebSearchEnabled()
    {
        return $this->webSearchEnabled;
    }

    /**
     * Get current model
     */
    public function getModel()
    {
        return $this->model;
    }
}
