<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Political parties table
        Schema::create('parties', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->string('short_name', 20)->unique();
            $table->string('color', 7)->nullable(); // Hex color
            $table->string('logo_url')->nullable();
            $table->text('description')->nullable();
            $table->year('founded_year')->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestamps();
        });

        // Regions table (16 Ghana regions)
        Schema::create('regions', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->string('code', 10)->unique();
            $table->integer('population')->nullable();
            $table->integer('registered_voters')->nullable();
            $table->timestamps();
        });

        // Constituencies table
        Schema::create('constituencies', function (Blueprint $table) {
            $table->id();
            $table->foreignId('region_id')->constrained()->onDelete('cascade');
            $table->string('name');
            $table->string('code', 20)->unique()->nullable();
            $table->integer('registered_voters')->nullable();
            $table->timestamps();
        });

        // Elections table
        Schema::create('elections', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->enum('type', ['presidential', 'parliamentary', 'presidential_runoff']);
            $table->year('year');
            $table->date('election_date')->nullable();
            $table->enum('status', ['upcoming', 'ongoing', 'completed'])->default('completed');
            $table->integer('total_registered_voters')->nullable();
            $table->integer('total_votes_cast')->nullable();
            $table->decimal('voter_turnout', 5, 2)->nullable();
            $table->integer('total_valid_votes')->nullable();
            $table->integer('total_rejected_votes')->nullable();
            $table->text('notes')->nullable();
            $table->timestamps();
            
            $table->unique(['type', 'year']);
        });

        // Candidates table
        Schema::create('candidates', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->foreignId('party_id')->nullable()->constrained()->onDelete('set null');
            $table->string('photo_url')->nullable();
            $table->text('biography')->nullable();
            $table->date('date_of_birth')->nullable();
            $table->string('position')->nullable(); // President, Vice President, MP
            $table->boolean('is_active')->default(true);
            $table->timestamps();
        });

        // Election candidates (links candidates to specific elections)
        Schema::create('election_candidates', function (Blueprint $table) {
            $table->id();
            $table->foreignId('election_id')->constrained()->onDelete('cascade');
            $table->foreignId('candidate_id')->constrained()->onDelete('cascade');
            $table->foreignId('party_id')->nullable()->constrained()->onDelete('set null');
            $table->foreignId('constituency_id')->nullable()->constrained()->onDelete('set null');
            $table->string('running_mate')->nullable(); // For presidential elections
            $table->boolean('is_incumbent')->default(false);
            $table->timestamps();
            
            $table->unique(['election_id', 'candidate_id', 'constituency_id'], 'election_candidates_unique');
        });

        // National election results
        Schema::create('election_results_national', function (Blueprint $table) {
            $table->id();
            $table->foreignId('election_id')->constrained()->onDelete('cascade');
            $table->foreignId('candidate_id')->nullable()->constrained()->onDelete('set null');
            $table->foreignId('party_id')->nullable()->constrained()->onDelete('set null');
            $table->bigInteger('votes');
            $table->decimal('percentage', 5, 2);
            $table->boolean('is_winner')->default(false);
            $table->timestamps();
            
            $table->unique(['election_id', 'candidate_id'], 'election_national_unique');
        });

        // Regional election results
        Schema::create('election_results_regional', function (Blueprint $table) {
            $table->id();
            $table->foreignId('election_id')->constrained()->onDelete('cascade');
            $table->foreignId('region_id')->constrained()->onDelete('cascade');
            $table->foreignId('candidate_id')->nullable()->constrained()->onDelete('set null');
            $table->foreignId('party_id')->nullable()->constrained()->onDelete('set null');
            $table->bigInteger('votes');
            $table->decimal('percentage', 5, 2);
            $table->bigInteger('total_votes_in_region')->nullable();
            $table->boolean('is_region_winner')->default(false);
            $table->timestamps();
            
            $table->unique(['election_id', 'region_id', 'party_id'], 'election_regional_unique');
        });

        // Constituency election results
        Schema::create('election_results_constituency', function (Blueprint $table) {
            $table->id();
            $table->foreignId('election_id')->constrained()->onDelete('cascade');
            $table->foreignId('constituency_id')->constrained()->onDelete('cascade');
            $table->foreignId('candidate_id')->nullable()->constrained()->onDelete('set null');
            $table->foreignId('party_id')->nullable()->constrained()->onDelete('set null');
            $table->bigInteger('votes');
            $table->decimal('percentage', 5, 2);
            $table->bigInteger('total_votes_in_constituency')->nullable();
            $table->boolean('is_constituency_winner')->default(false);
            $table->timestamps();
            
            $table->index(['election_id', 'constituency_id']);
        });

        // Parliamentary seats summary
        Schema::create('parliamentary_seats', function (Blueprint $table) {
            $table->id();
            $table->foreignId('election_id')->constrained()->onDelete('cascade');
            $table->foreignId('party_id')->nullable()->constrained()->onDelete('set null');
            $table->integer('seats_won');
            $table->decimal('percentage', 5, 2)->nullable();
            $table->timestamps();
            
            $table->unique(['election_id', 'party_id'], 'parliamentary_seats_unique');
        });

        // Polls table
        Schema::create('polls', function (Blueprint $table) {
            $table->id();
            $table->foreignId('election_id')->nullable()->constrained()->onDelete('set null');
            $table->string('pollster_name');
            $table->string('pollster_organization')->nullable();
            $table->text('methodology')->nullable();
            $table->integer('sample_size')->nullable();
            $table->decimal('margin_of_error', 4, 2)->nullable();
            $table->date('date_conducted');
            $table->date('date_published')->nullable();
            $table->string('source_url')->nullable();
            $table->text('source_document')->nullable(); // For uploaded PDFs
            $table->boolean('is_verified')->default(false);
            $table->text('notes')->nullable();
            $table->text('disclaimer')->nullable();
            $table->foreignId('uploaded_by')->nullable()->constrained('users')->onDelete('set null');
            $table->timestamps();
        });

        // Poll results
        Schema::create('poll_results', function (Blueprint $table) {
            $table->id();
            $table->foreignId('poll_id')->constrained()->onDelete('cascade');
            $table->foreignId('candidate_id')->nullable()->constrained()->onDelete('set null');
            $table->foreignId('party_id')->nullable()->constrained()->onDelete('set null');
            $table->decimal('percentage', 5, 2);
            $table->decimal('lower_bound', 5, 2)->nullable(); // For confidence intervals
            $table->decimal('upper_bound', 5, 2)->nullable();
            $table->timestamps();
        });

        // Social mentions (for X/Twitter sentiment tracking)
        Schema::create('social_mentions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('election_id')->nullable()->constrained()->onDelete('set null');
            $table->foreignId('candidate_id')->nullable()->constrained()->onDelete('set null');
            $table->foreignId('party_id')->nullable()->constrained()->onDelete('set null');
            $table->enum('platform', ['twitter', 'facebook', 'instagram', 'tiktok', 'other'])->default('twitter');
            $table->text('content_snippet')->nullable();
            $table->string('post_url')->nullable();
            $table->string('post_id')->nullable();
            $table->enum('sentiment', ['positive', 'negative', 'neutral'])->default('neutral');
            $table->decimal('sentiment_score', 4, 3)->nullable(); // -1 to 1
            $table->integer('engagement_score')->nullable(); // likes + retweets + replies
            $table->integer('likes')->nullable();
            $table->integer('retweets')->nullable();
            $table->integer('replies')->nullable();
            $table->string('hashtags')->nullable();
            $table->string('author_handle')->nullable();
            $table->integer('author_followers')->nullable();
            $table->timestamp('posted_at')->nullable();
            $table->timestamp('captured_at')->nullable();
            $table->timestamps();
            
            $table->index(['election_id', 'platform', 'sentiment']);
            $table->index(['candidate_id', 'posted_at']);
        });

        // AI predictions
        Schema::create('ai_predictions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('election_id')->constrained()->onDelete('cascade');
            $table->foreignId('candidate_id')->nullable()->constrained()->onDelete('set null');
            $table->foreignId('party_id')->nullable()->constrained()->onDelete('set null');
            $table->decimal('win_probability', 5, 2)->nullable(); // 0-100%
            $table->decimal('vote_share_prediction', 5, 2)->nullable();
            $table->decimal('lower_confidence', 5, 2)->nullable();
            $table->decimal('upper_confidence', 5, 2)->nullable();
            $table->text('qualitative_analysis')->nullable();
            $table->text('key_factors')->nullable(); // JSON array of factors
            $table->text('risks_and_caveats')->nullable();
            $table->json('data_sources')->nullable(); // References to polls, historical data, etc.
            $table->string('model_version')->nullable();
            $table->timestamp('generated_at');
            $table->timestamps();
            
            $table->index(['election_id', 'generated_at']);
        });

        // Election bookmarks (user-specific)
        Schema::create('election_bookmarks', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->foreignId('election_id')->nullable()->constrained()->onDelete('cascade');
            $table->foreignId('candidate_id')->nullable()->constrained()->onDelete('cascade');
            $table->foreignId('region_id')->nullable()->constrained()->onDelete('cascade');
            $table->text('notes')->nullable();
            $table->timestamps();
            
            $table->index(['user_id', 'election_id']);
        });

        // Election chat logs (separate from main chatbot)
        Schema::create('election_chat_logs', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->nullable()->constrained()->onDelete('set null');
            $table->string('session_id');
            $table->text('query');
            $table->text('response')->nullable();
            $table->string('intent')->nullable(); // prediction, comparison, historical, poll_check, etc.
            $table->json('context_data')->nullable();
            $table->json('data_sources_used')->nullable();
            $table->integer('response_time_ms')->nullable();
            $table->boolean('was_successful')->default(true);
            $table->string('error_message')->nullable();
            $table->string('user_ip')->nullable();
            $table->string('user_agent')->nullable();
            $table->timestamps();
            
            $table->index(['session_id', 'created_at']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('election_chat_logs');
        Schema::dropIfExists('election_bookmarks');
        Schema::dropIfExists('ai_predictions');
        Schema::dropIfExists('social_mentions');
        Schema::dropIfExists('poll_results');
        Schema::dropIfExists('polls');
        Schema::dropIfExists('parliamentary_seats');
        Schema::dropIfExists('election_results_constituency');
        Schema::dropIfExists('election_results_regional');
        Schema::dropIfExists('election_results_national');
        Schema::dropIfExists('election_candidates');
        Schema::dropIfExists('candidates');
        Schema::dropIfExists('elections');
        Schema::dropIfExists('constituencies');
        Schema::dropIfExists('regions');
        Schema::dropIfExists('parties');
    }
};
