<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class ElectionDataSeeder extends Seeder
{
    /**
     * Seed the election database with historical Ghana election data.
     */
    public function run(): void
    {
        // Seed parties
        $this->seedParties();
        
        // Seed regions
        $this->seedRegions();
        
        // Seed elections
        $this->seedElections();
        
        // Seed candidates
        $this->seedCandidates();
        
        // Seed historical election results
        $this->seedElectionResults();
        
        // Seed parliamentary seats
        $this->seedParliamentarySeats();

        // Overwrite regional results from Election Data.xlsx (election_history) when file exists
        $excelPath = base_path('election_history/Election Data.xlsx');
        if (is_file($excelPath)) {
            \Illuminate\Support\Facades\Artisan::call('election:import-excel', ['file' => 'election_history/Election Data.xlsx']);
        }
    }

    private function seedParties(): void
    {
        $parties = [
            [
                'name' => 'New Patriotic Party',
                'short_name' => 'NPP',
                'color' => '#0057B7', // Blue
                'description' => 'Centre-right political party in Ghana, founded in 1992.',
                'founded_year' => 1992,
                'is_active' => true,
            ],
            [
                'name' => 'National Democratic Congress',
                'short_name' => 'NDC',
                'color' => '#009739', // Green
                'description' => 'Centre-left political party in Ghana, founded in 1992.',
                'founded_year' => 1992,
                'is_active' => true,
            ],
            [
                'name' => 'Convention People\'s Party',
                'short_name' => 'CPP',
                'color' => '#FF0000', // Red
                'description' => 'Socialist political party, founded by Kwame Nkrumah.',
                'founded_year' => 1949,
                'is_active' => true,
            ],
            [
                'name' => 'Progressive People\'s Party',
                'short_name' => 'PPP',
                'color' => '#FFD700', // Gold
                'description' => 'Liberal political party in Ghana.',
                'founded_year' => 2012,
                'is_active' => true,
            ],
            [
                'name' => 'Independent',
                'short_name' => 'IND',
                'color' => '#808080', // Gray
                'description' => 'Independent candidates not affiliated with any party.',
                'founded_year' => null,
                'is_active' => true,
            ],
            [
                'name' => 'Others',
                'short_name' => 'OTH',
                'color' => '#A0A0A0', // Light gray
                'description' => 'Other minor parties and candidates.',
                'founded_year' => null,
                'is_active' => true,
            ],
        ];

        foreach ($parties as $party) {
            DB::table('parties')->updateOrInsert(
                ['short_name' => $party['short_name']],
                array_merge($party, ['created_at' => now(), 'updated_at' => now()])
            );
        }
    }

    private function seedRegions(): void
    {
        // 2021 Ghana Population and Housing Census - Ghana Statistical Service
        $regions = [
            ['name' => 'Ahafo', 'code' => 'AH', 'population' => 564668],
            ['name' => 'Ashanti', 'code' => 'AS', 'population' => 5440463],
            ['name' => 'Bono', 'code' => 'BO', 'population' => 1208649],
            ['name' => 'Bono East', 'code' => 'BE', 'population' => 1203400],
            ['name' => 'Central', 'code' => 'CE', 'population' => 2859821],
            ['name' => 'Eastern', 'code' => 'EA', 'population' => 2925653],
            ['name' => 'Greater Accra', 'code' => 'GA', 'population' => 5445692],
            ['name' => 'North East', 'code' => 'NE', 'population' => 658946],
            ['name' => 'Northern', 'code' => 'NO', 'population' => 2310939],
            ['name' => 'Oti', 'code' => 'OT', 'population' => 747248],
            ['name' => 'Savannah', 'code' => 'SA', 'population' => 653266],
            ['name' => 'Upper East', 'code' => 'UE', 'population' => 1301226],
            ['name' => 'Upper West', 'code' => 'UW', 'population' => 901502],
            ['name' => 'Volta', 'code' => 'VO', 'population' => 1659040],
            ['name' => 'Western', 'code' => 'WE', 'population' => 2060585],
            ['name' => 'Western North', 'code' => 'WN', 'population' => 880921],
        ];

        foreach ($regions as $region) {
            DB::table('regions')->updateOrInsert(
                ['code' => $region['code']],
                array_merge($region, ['created_at' => now(), 'updated_at' => now()])
            );
        }
    }

    private function seedElections(): void
    {
        $elections = [
            ['name' => '1996 Presidential Election', 'type' => 'presidential', 'year' => 1996, 'election_date' => '1996-12-07', 'status' => 'completed'],
            ['name' => '2000 Presidential Election', 'type' => 'presidential', 'year' => 2000, 'election_date' => '2000-12-07', 'status' => 'completed'],
            ['name' => '2000 Presidential Runoff', 'type' => 'presidential_runoff', 'year' => 2000, 'election_date' => '2000-12-28', 'status' => 'completed'],
            ['name' => '2004 Presidential Election', 'type' => 'presidential', 'year' => 2004, 'election_date' => '2004-12-07', 'status' => 'completed'],
            ['name' => '2008 Presidential Election', 'type' => 'presidential', 'year' => 2008, 'election_date' => '2008-12-07', 'status' => 'completed'],
            ['name' => '2008 Presidential Runoff', 'type' => 'presidential_runoff', 'year' => 2008, 'election_date' => '2008-12-28', 'status' => 'completed'],
            ['name' => '2012 Presidential Election', 'type' => 'presidential', 'year' => 2012, 'election_date' => '2012-12-07', 'status' => 'completed'],
            ['name' => '2016 Presidential Election', 'type' => 'presidential', 'year' => 2016, 'election_date' => '2016-12-07', 'status' => 'completed'],
            ['name' => '2020 Presidential Election', 'type' => 'presidential', 'year' => 2020, 'election_date' => '2020-12-07', 'status' => 'completed'],
            ['name' => '2024 Presidential Election', 'type' => 'presidential', 'year' => 2024, 'election_date' => '2024-12-07', 'status' => 'completed'],
            // Parliamentary elections
            ['name' => '2000 Parliamentary Election', 'type' => 'parliamentary', 'year' => 2000, 'election_date' => '2000-12-07', 'status' => 'completed'],
            ['name' => '2004 Parliamentary Election', 'type' => 'parliamentary', 'year' => 2004, 'election_date' => '2004-12-07', 'status' => 'completed'],
            ['name' => '2008 Parliamentary Election', 'type' => 'parliamentary', 'year' => 2008, 'election_date' => '2008-12-07', 'status' => 'completed'],
            ['name' => '2012 Parliamentary Election', 'type' => 'parliamentary', 'year' => 2012, 'election_date' => '2012-12-07', 'status' => 'completed'],
            ['name' => '2016 Parliamentary Election', 'type' => 'parliamentary', 'year' => 2016, 'election_date' => '2016-12-07', 'status' => 'completed'],
            ['name' => '2020 Parliamentary Election', 'type' => 'parliamentary', 'year' => 2020, 'election_date' => '2020-12-07', 'status' => 'completed'],
            ['name' => '2024 Parliamentary Election', 'type' => 'parliamentary', 'year' => 2024, 'election_date' => '2024-12-07', 'status' => 'completed'],
        ];

        foreach ($elections as $election) {
            DB::table('elections')->updateOrInsert(
                ['type' => $election['type'], 'year' => $election['year']],
                array_merge($election, ['created_at' => now(), 'updated_at' => now()])
            );
        }
    }

    private function seedCandidates(): void
    {
        $candidates = [
            // NPP Candidates
            ['name' => 'John Agyekum Kuffour', 'party' => 'NPP', 'position' => 'President'],
            ['name' => 'Nana Akufo-Addo', 'party' => 'NPP', 'position' => 'President'],
            ['name' => 'Dr. Mahamudu Bawumia', 'party' => 'NPP', 'position' => 'President'],
            
            // NDC Candidates
            ['name' => 'John Atta Mills', 'party' => 'NDC', 'position' => 'President'],
            ['name' => 'John Dramani Mahama', 'party' => 'NDC', 'position' => 'President'],
            ['name' => 'Jerry John Rawlings', 'party' => 'NDC', 'position' => 'President'],
        ];

        foreach ($candidates as $candidate) {
            $partyId = DB::table('parties')->where('short_name', $candidate['party'])->value('id');
            
            DB::table('candidates')->updateOrInsert(
                ['name' => $candidate['name']],
                [
                    'name' => $candidate['name'],
                    'party_id' => $partyId,
                    'position' => $candidate['position'],
                    'is_active' => true,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]
            );
        }
    }

    private function seedElectionResults(): void
    {
        // Historical Presidential Election Results by Region
        // Data from Ghana Electoral Commission / PeaceFM Online
        
        $results = [
            // 2000 Presidential Election (First Round)
            ['year' => 2000, 'type' => 'presidential', 'region' => 'Ahafo', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 77779, 'npp_pct' => 56.69, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 55797, 'ndc_pct' => 40.66, 'others_votes' => 5673, 'others_pct' => 2.65, 'total' => 139249, 'winner' => 'NPP'],
            ['year' => 2000, 'type' => 'presidential', 'region' => 'Ashanti', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 950597, 'npp_pct' => 74.80, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 286012, 'ndc_pct' => 22.51, 'others_votes' => 52757, 'others_pct' => 2.69, 'total' => 1289366, 'winner' => 'NPP'],
            ['year' => 2000, 'type' => 'presidential', 'region' => 'Bono', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 101595, 'npp_pct' => 62.36, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 55746, 'ndc_pct' => 34.22, 'others_votes' => 7785, 'others_pct' => 3.42, 'total' => 165126, 'winner' => 'NPP'],
            ['year' => 2000, 'type' => 'presidential', 'region' => 'Bono East', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 33910, 'npp_pct' => 41.93, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 43015, 'ndc_pct' => 53.19, 'others_votes' => 4850, 'others_pct' => 4.88, 'total' => 81775, 'winner' => 'NDC'],
            ['year' => 2000, 'type' => 'presidential', 'region' => 'Central', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 259367, 'npp_pct' => 49.67, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 227234, 'ndc_pct' => 43.51, 'others_votes' => 74639, 'others_pct' => 6.82, 'total' => 561240, 'winner' => 'NPP'],
            ['year' => 2000, 'type' => 'presidential', 'region' => 'Eastern', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 353398, 'npp_pct' => 52.84, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 290374, 'ndc_pct' => 43.42, 'others_votes' => 76276, 'others_pct' => 3.74, 'total' => 720048, 'winner' => 'NPP'],
            ['year' => 2000, 'type' => 'presidential', 'region' => 'Greater Accra', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 580163, 'npp_pct' => 53.10, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 459884, 'ndc_pct' => 42.09, 'others_votes' => 76182, 'others_pct' => 4.81, 'total' => 1116229, 'winner' => 'NPP'],
            ['year' => 2000, 'type' => 'presidential', 'region' => 'North East', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 11613, 'npp_pct' => 11.29, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 46353, 'ndc_pct' => 45.05, 'others_votes' => 51573, 'others_pct' => 43.66, 'total' => 109539, 'winner' => 'NDC'],
            ['year' => 2000, 'type' => 'presidential', 'region' => 'Northern', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 127568, 'npp_pct' => 34.30, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 188755, 'ndc_pct' => 50.75, 'others_votes' => 74030, 'others_pct' => 14.95, 'total' => 390353, 'winner' => 'NDC'],
            ['year' => 2000, 'type' => 'presidential', 'region' => 'Oti', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 10828, 'npp_pct' => 17.17, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 47940, 'ndc_pct' => 76.02, 'others_votes' => 6026, 'others_pct' => 6.81, 'total' => 64794, 'winner' => 'NDC'],
            ['year' => 2000, 'type' => 'presidential', 'region' => 'Savannah', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 27646, 'npp_pct' => 30.97, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 51240, 'ndc_pct' => 57.40, 'others_votes' => 20424, 'others_pct' => 11.63, 'total' => 99310, 'winner' => 'NDC'],
            ['year' => 2000, 'type' => 'presidential', 'region' => 'Upper East', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 53225, 'npp_pct' => 18.99, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 139665, 'ndc_pct' => 49.82, 'others_votes' => 116358, 'others_pct' => 31.19, 'total' => 309248, 'winner' => 'NDC'],
            ['year' => 2000, 'type' => 'presidential', 'region' => 'Upper West', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 27714, 'npp_pct' => 15.51, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 111345, 'ndc_pct' => 62.29, 'others_votes' => 50826, 'others_pct' => 22.20, 'total' => 189885, 'winner' => 'NDC'],
            ['year' => 2000, 'type' => 'presidential', 'region' => 'Volta', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 35444, 'npp_pct' => 6.78, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 457674, 'ndc_pct' => 87.51, 'others_votes' => 51078, 'others_pct' => 5.71, 'total' => 544196, 'winner' => 'NDC'],
            ['year' => 2000, 'type' => 'presidential', 'region' => 'Western', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 243909, 'npp_pct' => 53.04, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 185792, 'ndc_pct' => 40.41, 'others_votes' => 41979, 'others_pct' => 6.55, 'total' => 471680, 'winner' => 'NPP'],
            ['year' => 2000, 'type' => 'presidential', 'region' => 'Western North', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 52946, 'npp_pct' => 32.28, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 105207, 'ndc_pct' => 64.15, 'others_votes' => 6774, 'others_pct' => 3.57, 'total' => 164927, 'winner' => 'NDC'],

            // 2000 Presidential Runoff
            ['year' => 2000, 'type' => 'presidential_runoff', 'region' => 'Ahafo', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 84495, 'npp_pct' => 62.59, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 50493, 'ndc_pct' => 37.41, 'others_votes' => 0, 'others_pct' => 0, 'total' => 134988, 'winner' => 'NPP'],
            ['year' => 2000, 'type' => 'presidential_runoff', 'region' => 'Ashanti', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 972170, 'npp_pct' => 80.50, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 235502, 'ndc_pct' => 19.50, 'others_votes' => 0, 'others_pct' => 0, 'total' => 1207672, 'winner' => 'NPP'],
            ['year' => 2000, 'type' => 'presidential_runoff', 'region' => 'Bono', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 108413, 'npp_pct' => 69.73, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 47056, 'ndc_pct' => 30.27, 'others_votes' => 0, 'others_pct' => 0, 'total' => 155469, 'winner' => 'NPP'],
            ['year' => 2000, 'type' => 'presidential_runoff', 'region' => 'Bono East', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 32392, 'npp_pct' => 56.72, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 24717, 'ndc_pct' => 43.28, 'others_votes' => 0, 'others_pct' => 0, 'total' => 57109, 'winner' => 'NPP'],
            ['year' => 2000, 'type' => 'presidential_runoff', 'region' => 'Central', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 301139, 'npp_pct' => 60.25, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 198682, 'ndc_pct' => 39.75, 'others_votes' => 0, 'others_pct' => 0, 'total' => 499821, 'winner' => 'NPP'],
            ['year' => 2000, 'type' => 'presidential_runoff', 'region' => 'Eastern', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 459259, 'npp_pct' => 63.50, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 264024, 'ndc_pct' => 36.50, 'others_votes' => 0, 'others_pct' => 0, 'total' => 723283, 'winner' => 'NPP'],
            ['year' => 2000, 'type' => 'presidential_runoff', 'region' => 'Greater Accra', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 616729, 'npp_pct' => 60.44, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 403725, 'ndc_pct' => 39.56, 'others_votes' => 0, 'others_pct' => 0, 'total' => 1020454, 'winner' => 'NPP'],
            ['year' => 2000, 'type' => 'presidential_runoff', 'region' => 'North East', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 30454, 'npp_pct' => 48.35, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 32529, 'ndc_pct' => 51.65, 'others_votes' => 0, 'others_pct' => 0, 'total' => 62983, 'winner' => 'NDC'],
            ['year' => 2000, 'type' => 'presidential_runoff', 'region' => 'Northern', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 179227, 'npp_pct' => 49.54, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 182576, 'ndc_pct' => 50.46, 'others_votes' => 0, 'others_pct' => 0, 'total' => 361803, 'winner' => 'NDC'],
            ['year' => 2000, 'type' => 'presidential_runoff', 'region' => 'Oti', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 13859, 'npp_pct' => 22.76, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 47046, 'ndc_pct' => 77.24, 'others_votes' => 0, 'others_pct' => 0, 'total' => 60905, 'winner' => 'NDC'],
            ['year' => 2000, 'type' => 'presidential_runoff', 'region' => 'Savannah', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 29367, 'npp_pct' => 43.47, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 38191, 'ndc_pct' => 56.53, 'others_votes' => 0, 'others_pct' => 0, 'total' => 67558, 'winner' => 'NDC'],
            ['year' => 2000, 'type' => 'presidential_runoff', 'region' => 'Upper East', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 111987, 'npp_pct' => 41.63, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 156997, 'ndc_pct' => 58.37, 'others_votes' => 0, 'others_pct' => 0, 'total' => 268984, 'winner' => 'NDC'],
            ['year' => 2000, 'type' => 'presidential_runoff', 'region' => 'Upper West', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 64165, 'npp_pct' => 38.04, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 104534, 'ndc_pct' => 61.96, 'others_votes' => 0, 'others_pct' => 0, 'total' => 168699, 'winner' => 'NDC'],
            ['year' => 2000, 'type' => 'presidential_runoff', 'region' => 'Volta', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 54111, 'npp_pct' => 9.53, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 513883, 'ndc_pct' => 90.47, 'others_votes' => 0, 'others_pct' => 0, 'total' => 567994, 'winner' => 'NDC'],
            ['year' => 2000, 'type' => 'presidential_runoff', 'region' => 'Western', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 187641, 'npp_pct' => 68.03, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 88247, 'ndc_pct' => 31.97, 'others_votes' => 0, 'others_pct' => 0, 'total' => 275888, 'winner' => 'NPP'],
            ['year' => 2000, 'type' => 'presidential_runoff', 'region' => 'Western North', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 50532, 'npp_pct' => 42.34, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 68817, 'ndc_pct' => 57.66, 'others_votes' => 0, 'others_pct' => 0, 'total' => 119349, 'winner' => 'NDC'],

            // 2004 Presidential Election
            ['year' => 2004, 'type' => 'presidential', 'region' => 'Ahafo', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 94579, 'npp_pct' => 57.41, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 67838, 'ndc_pct' => 41.18, 'others_votes' => 6999, 'others_pct' => 1.41, 'total' => 169416, 'winner' => 'NPP'],
            ['year' => 2004, 'type' => 'presidential', 'region' => 'Ashanti', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 1404336, 'npp_pct' => 76.97, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 398572, 'ndc_pct' => 21.84, 'others_votes' => 0, 'others_pct' => 0, 'total' => 1802908, 'winner' => 'NPP'],
            ['year' => 2004, 'type' => 'presidential', 'region' => 'Bono', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 188087, 'npp_pct' => 58.42, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 128185, 'ndc_pct' => 39.81, 'others_votes' => 13329, 'others_pct' => 1.77, 'total' => 329601, 'winner' => 'NPP'],
            ['year' => 2004, 'type' => 'presidential', 'region' => 'Bono East', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 110367, 'npp_pct' => 40.91, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 152782, 'ndc_pct' => 56.63, 'others_votes' => 26123, 'others_pct' => 2.46, 'total' => 289272, 'winner' => 'NDC'],
            ['year' => 2004, 'type' => 'presidential', 'region' => 'Central', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 415813, 'npp_pct' => 58.80, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 275415, 'ndc_pct' => 38.94, 'others_votes' => 40036, 'others_pct' => 2.26, 'total' => 731264, 'winner' => 'NPP'],
            ['year' => 2004, 'type' => 'presidential', 'region' => 'Eastern', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 554627, 'npp_pct' => 60.30, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 353032, 'ndc_pct' => 38.38, 'others_votes' => 30945, 'others_pct' => 1.32, 'total' => 938604, 'winner' => 'NPP'],
            ['year' => 2004, 'type' => 'presidential', 'region' => 'Greater Accra', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 900159, 'npp_pct' => 51.71, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 811863, 'ndc_pct' => 46.64, 'others_votes' => 68898, 'others_pct' => 1.65, 'total' => 1780920, 'winner' => 'NPP'],
            ['year' => 2004, 'type' => 'presidential', 'region' => 'North East', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 38688, 'npp_pct' => 32.22, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 51573, 'ndc_pct' => 42.96, 'others_votes' => 40199, 'others_pct' => 24.82, 'total' => 130460, 'winner' => 'NDC'],
            ['year' => 2004, 'type' => 'presidential', 'region' => 'Northern', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 162093, 'npp_pct' => 35.56, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 278679, 'ndc_pct' => 61.14, 'others_votes' => 94782, 'others_pct' => 3.30, 'total' => 535554, 'winner' => 'NDC'],
            ['year' => 2004, 'type' => 'presidential', 'region' => 'Oti', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 50564, 'npp_pct' => 28.45, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 116216, 'ndc_pct' => 65.39, 'others_votes' => 17176, 'others_pct' => 6.16, 'total' => 183956, 'winner' => 'NDC'],
            ['year' => 2004, 'type' => 'presidential', 'region' => 'Savannah', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 31731, 'npp_pct' => 33.81, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 57021, 'ndc_pct' => 60.76, 'others_votes' => 43371, 'others_pct' => 5.43, 'total' => 132123, 'winner' => 'NDC'],
            ['year' => 2004, 'type' => 'presidential', 'region' => 'Upper East', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 105003, 'npp_pct' => 31.69, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 176491, 'ndc_pct' => 53.26, 'others_votes' => 67324, 'others_pct' => 15.05, 'total' => 348818, 'winner' => 'NDC'],
            ['year' => 2004, 'type' => 'presidential', 'region' => 'Upper West', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 68238, 'npp_pct' => 32.22, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 119982, 'ndc_pct' => 56.65, 'others_votes' => 35640, 'others_pct' => 11.13, 'total' => 223860, 'winner' => 'NDC'],
            ['year' => 2004, 'type' => 'presidential', 'region' => 'Volta', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 50143, 'npp_pct' => 8.98, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 502827, 'ndc_pct' => 90.00, 'others_votes' => 0, 'others_pct' => 0, 'total' => 552970, 'winner' => 'NDC'],
            ['year' => 2004, 'type' => 'presidential', 'region' => 'Western', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 379024, 'npp_pct' => 66.39, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 176797, 'ndc_pct' => 30.97, 'others_votes' => 21064, 'others_pct' => 2.64, 'total' => 576885, 'winner' => 'NPP'],
            ['year' => 2004, 'type' => 'presidential', 'region' => 'Western North', 'npp_candidate' => 'John Agyekum Kuffour', 'npp_votes' => 104862, 'npp_pct' => 39.17, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 158340, 'ndc_pct' => 59.14, 'others_votes' => 13338, 'others_pct' => 1.69, 'total' => 276540, 'winner' => 'NDC'],

            // 2008 Presidential Election (First Round)
            ['year' => 2008, 'type' => 'presidential', 'region' => 'Ahafo', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 96795, 'npp_pct' => 54.59, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 78145, 'ndc_pct' => 44.07, 'others_votes' => 6820, 'others_pct' => 1.34, 'total' => 181760, 'winner' => 'NPP'],
            ['year' => 2008, 'type' => 'presidential', 'region' => 'Ashanti', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 1219530, 'npp_pct' => 72.53, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 437243, 'ndc_pct' => 26.01, 'others_votes' => 48668, 'others_pct' => 1.46, 'total' => 1705441, 'winner' => 'NPP'],
            ['year' => 2008, 'type' => 'presidential', 'region' => 'Bono', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 177032, 'npp_pct' => 54.60, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 141703, 'ndc_pct' => 43.71, 'others_votes' => 12495, 'others_pct' => 1.69, 'total' => 331230, 'winner' => 'NPP'],
            ['year' => 2008, 'type' => 'presidential', 'region' => 'Bono East', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 118761, 'npp_pct' => 43.20, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 150556, 'ndc_pct' => 54.76, 'others_votes' => 14888, 'others_pct' => 2.04, 'total' => 284205, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential', 'region' => 'Central', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 313665, 'npp_pct' => 45.97, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 345126, 'ndc_pct' => 50.58, 'others_votes' => 46437, 'others_pct' => 3.45, 'total' => 705228, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential', 'region' => 'Eastern', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 491520, 'npp_pct' => 57.14, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 353522, 'ndc_pct' => 41.10, 'others_votes' => 35111, 'others_pct' => 1.76, 'total' => 880153, 'winner' => 'NPP'],
            ['year' => 2008, 'type' => 'presidential', 'region' => 'Greater Accra', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 768465, 'npp_pct' => 46.03, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 870011, 'ndc_pct' => 52.11, 'others_votes' => 48953, 'others_pct' => 1.86, 'total' => 1687429, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential', 'region' => 'North East', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 44697, 'npp_pct' => 35.00, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 68381, 'ndc_pct' => 53.54, 'others_votes' => 21529, 'others_pct' => 11.46, 'total' => 134607, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential', 'region' => 'Northern', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 208749, 'npp_pct' => 38.06, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 320464, 'ndc_pct' => 58.43, 'others_votes' => 25975, 'others_pct' => 3.51, 'total' => 555188, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential', 'region' => 'Oti', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 58673, 'npp_pct' => 34.14, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 108393, 'ndc_pct' => 63.07, 'others_votes' => 9980, 'others_pct' => 2.79, 'total' => 177046, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential', 'region' => 'Savannah', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 49880, 'npp_pct' => 39.47, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 71600, 'ndc_pct' => 56.65, 'others_votes' => 10838, 'others_pct' => 3.88, 'total' => 132318, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential', 'region' => 'Upper East', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 118454, 'npp_pct' => 35.25, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 188395, 'ndc_pct' => 56.06, 'others_votes' => 33417, 'others_pct' => 8.69, 'total' => 340266, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential', 'region' => 'Upper West', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 81137, 'npp_pct' => 37.72, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 116922, 'ndc_pct' => 54.36, 'others_votes' => 28481, 'others_pct' => 7.92, 'total' => 226540, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential', 'region' => 'Volta', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 43695, 'npp_pct' => 8.85, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 440606, 'ndc_pct' => 89.21, 'others_votes' => 17939, 'others_pct' => 1.94, 'total' => 502240, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential', 'region' => 'Western', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 315168, 'npp_pct' => 54.10, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 230403, 'ndc_pct' => 39.55, 'others_votes' => 10592, 'others_pct' => 6.35, 'total' => 556163, 'winner' => 'NPP'],
            ['year' => 2008, 'type' => 'presidential', 'region' => 'Western North', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 97834, 'npp_pct' => 38.86, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 149419, 'ndc_pct' => 59.34, 'others_votes' => 13787, 'others_pct' => 1.80, 'total' => 261040, 'winner' => 'NDC'],

            // 2008 Presidential Runoff
            ['year' => 2008, 'type' => 'presidential_runoff', 'region' => 'Ahafo', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 104572, 'npp_pct' => 54.91, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 85865, 'ndc_pct' => 45.09, 'others_votes' => 0, 'others_pct' => 0, 'total' => 190437, 'winner' => 'NPP'],
            ['year' => 2008, 'type' => 'presidential_runoff', 'region' => 'Ashanti', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 1393496, 'npp_pct' => 74.39, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 479633, 'ndc_pct' => 25.61, 'others_votes' => 0, 'others_pct' => 0, 'total' => 1873129, 'winner' => 'NPP'],
            ['year' => 2008, 'type' => 'presidential_runoff', 'region' => 'Bono', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 167355, 'npp_pct' => 51.87, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 155302, 'ndc_pct' => 48.13, 'others_votes' => 0, 'others_pct' => 0, 'total' => 322657, 'winner' => 'NPP'],
            ['year' => 2008, 'type' => 'presidential_runoff', 'region' => 'Bono East', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 118296, 'npp_pct' => 41.07, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 169717, 'ndc_pct' => 58.93, 'others_votes' => 0, 'others_pct' => 0, 'total' => 288013, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential_runoff', 'region' => 'Central', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 325454, 'npp_pct' => 46.20, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 378975, 'ndc_pct' => 53.80, 'others_votes' => 0, 'others_pct' => 0, 'total' => 704429, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential_runoff', 'region' => 'Eastern', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 539475, 'npp_pct' => 57.47, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 399278, 'ndc_pct' => 42.53, 'others_votes' => 0, 'others_pct' => 0, 'total' => 938753, 'winner' => 'NPP'],
            ['year' => 2008, 'type' => 'presidential_runoff', 'region' => 'Greater Accra', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 796541, 'npp_pct' => 45.54, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 952599, 'ndc_pct' => 54.46, 'others_votes' => 0, 'others_pct' => 0, 'total' => 1749140, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential_runoff', 'region' => 'North East', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 48700, 'npp_pct' => 37.91, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 79765, 'ndc_pct' => 62.09, 'others_votes' => 0, 'others_pct' => 0, 'total' => 128465, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential_runoff', 'region' => 'Northern', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 214646, 'npp_pct' => 38.44, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 343815, 'ndc_pct' => 61.56, 'others_votes' => 0, 'others_pct' => 0, 'total' => 558461, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential_runoff', 'region' => 'Oti', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 58028, 'npp_pct' => 33.35, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 116483, 'ndc_pct' => 66.75, 'others_votes' => 0, 'others_pct' => 0, 'total' => 174511, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential_runoff', 'region' => 'Savannah', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 49435, 'npp_pct' => 38.68, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 78381, 'ndc_pct' => 61.32, 'others_votes' => 0, 'others_pct' => 0, 'total' => 127816, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential_runoff', 'region' => 'Upper East', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 117477, 'npp_pct' => 34.40, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 223994, 'ndc_pct' => 65.60, 'others_votes' => 0, 'others_pct' => 0, 'total' => 341471, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential_runoff', 'region' => 'Upper West', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 81561, 'npp_pct' => 37.68, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 134917, 'ndc_pct' => 62.32, 'others_votes' => 0, 'others_pct' => 0, 'total' => 216478, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential_runoff', 'region' => 'Volta', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 44145, 'npp_pct' => 7.90, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 514416, 'ndc_pct' => 92.10, 'others_votes' => 0, 'others_pct' => 0, 'total' => 558561, 'winner' => 'NDC'],
            ['year' => 2008, 'type' => 'presidential_runoff', 'region' => 'Western', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 292669, 'npp_pct' => 53.45, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 254886, 'ndc_pct' => 46.55, 'others_votes' => 0, 'others_pct' => 0, 'total' => 547555, 'winner' => 'NPP'],
            ['year' => 2008, 'type' => 'presidential_runoff', 'region' => 'Western North', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 91806, 'npp_pct' => 36.41, 'ndc_candidate' => 'John Atta Mills', 'ndc_votes' => 160362, 'ndc_pct' => 63.59, 'others_votes' => 0, 'others_pct' => 0, 'total' => 252168, 'winner' => 'NDC'],

            // 2012 Presidential Election
            ['year' => 2012, 'type' => 'presidential', 'region' => 'Ahafo', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 115226, 'npp_pct' => 50.78, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 109600, 'ndc_pct' => 48.30, 'others_votes' => 7111, 'others_pct' => 3.06, 'total' => 231937, 'winner' => 'NPP'],
            ['year' => 2012, 'type' => 'presidential', 'region' => 'Ashanti', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 1531152, 'npp_pct' => 70.86, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 612616, 'ndc_pct' => 28.35, 'others_votes' => 46525, 'others_pct' => 0.79, 'total' => 2190293, 'winner' => 'NPP'],
            ['year' => 2012, 'type' => 'presidential', 'region' => 'Bono', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 216993, 'npp_pct' => 52.75, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 189731, 'ndc_pct' => 46.12, 'others_votes' => 14276, 'others_pct' => 1.13, 'total' => 421000, 'winner' => 'NPP'],
            ['year' => 2012, 'type' => 'presidential', 'region' => 'Bono East', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 137690, 'npp_pct' => 38.83, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 211913, 'ndc_pct' => 59.76, 'others_votes' => 17828, 'others_pct' => 1.41, 'total' => 367431, 'winner' => 'NDC'],
            ['year' => 2012, 'type' => 'presidential', 'region' => 'Central', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 430135, 'npp_pct' => 45.53, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 492374, 'ndc_pct' => 52.12, 'others_votes' => 48527, 'others_pct' => 2.35, 'total' => 971036, 'winner' => 'NDC'],
            ['year' => 2012, 'type' => 'presidential', 'region' => 'Eastern', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 655462, 'npp_pct' => 56.91, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 483998, 'ndc_pct' => 42.03, 'others_votes' => 24916, 'others_pct' => 1.06, 'total' => 1164376, 'winner' => 'NPP'],
            ['year' => 2012, 'type' => 'presidential', 'region' => 'Greater Accra', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 1009787, 'npp_pct' => 46.92, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 1125751, 'ndc_pct' => 52.31, 'others_votes' => 39110, 'others_pct' => 0.77, 'total' => 2174648, 'winner' => 'NDC'],
            ['year' => 2012, 'type' => 'presidential', 'region' => 'North East', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 73384, 'npp_pct' => 44.86, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 85283, 'ndc_pct' => 52.13, 'others_votes' => 11982, 'others_pct' => 3.01, 'total' => 170649, 'winner' => 'NDC'],
            ['year' => 2012, 'type' => 'presidential', 'region' => 'Northern', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 255454, 'npp_pct' => 38.78, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 385906, 'ndc_pct' => 58.59, 'others_votes' => 38271, 'others_pct' => 2.63, 'total' => 679631, 'winner' => 'NDC'],
            ['year' => 2012, 'type' => 'presidential', 'region' => 'Oti', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 64207, 'npp_pct' => 28.45, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 155343, 'ndc_pct' => 68.83, 'others_votes' => 16145, 'others_pct' => 2.72, 'total' => 235695, 'winner' => 'NDC'],
            ['year' => 2012, 'type' => 'presidential', 'region' => 'Savannah', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 99413, 'npp_pct' => 63.04, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 54425, 'ndc_pct' => 34.51, 'others_votes' => 10838, 'others_pct' => 2.45, 'total' => 164676, 'winner' => 'NPP'],
            ['year' => 2012, 'type' => 'presidential', 'region' => 'Upper East', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 120814, 'npp_pct' => 29.29, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 274019, 'ndc_pct' => 66.43, 'others_votes' => 37215, 'others_pct' => 4.28, 'total' => 432048, 'winner' => 'NDC'],
            ['year' => 2012, 'type' => 'presidential', 'region' => 'Upper West', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 83098, 'npp_pct' => 29.26, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 186134, 'ndc_pct' => 65.54, 'others_votes' => 28903, 'others_pct' => 5.20, 'total' => 298135, 'winner' => 'NDC'],
            ['year' => 2012, 'type' => 'presidential', 'region' => 'Volta', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 46942, 'npp_pct' => 7.41, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 579298, 'ndc_pct' => 91.39, 'others_votes' => 21804, 'others_pct' => 1.20, 'total' => 648044, 'winner' => 'NDC'],
            ['year' => 2012, 'type' => 'presidential', 'region' => 'Western', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 342217, 'npp_pct' => 47.69, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 360542, 'ndc_pct' => 50.24, 'others_votes' => 32370, 'others_pct' => 2.00, 'total' => 735129, 'winner' => 'NDC'],
            ['year' => 2012, 'type' => 'presidential', 'region' => 'Western North', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 221651, 'npp_pct' => 62.95, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 126300, 'ndc_pct' => 35.87, 'others_votes' => 15901, 'others_pct' => 1.18, 'total' => 363852, 'winner' => 'NPP'],

            // 2016 Presidential Election
            ['year' => 2016, 'type' => 'presidential', 'region' => 'Ahafo', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 123139, 'npp_pct' => 55.11, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 98272, 'ndc_pct' => 43.98, 'others_votes' => 5108, 'others_pct' => 0.91, 'total' => 226519, 'winner' => 'NPP'],
            ['year' => 2016, 'type' => 'presidential', 'region' => 'Ashanti', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 1646949, 'npp_pct' => 76.27, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 497242, 'ndc_pct' => 23.03, 'others_votes' => 26064, 'others_pct' => 0.70, 'total' => 2170255, 'winner' => 'NPP'],
            ['year' => 2016, 'type' => 'presidential', 'region' => 'Bono', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 243904, 'npp_pct' => 60.30, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 155488, 'ndc_pct' => 38.44, 'others_votes' => 11681, 'others_pct' => 1.26, 'total' => 411073, 'winner' => 'NPP'],
            ['year' => 2016, 'type' => 'presidential', 'region' => 'Bono East', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 164104, 'npp_pct' => 45.84, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 189029, 'ndc_pct' => 52.80, 'others_votes' => 12399, 'others_pct' => 1.36, 'total' => 365532, 'winner' => 'NDC'],
            ['year' => 2016, 'type' => 'presidential', 'region' => 'Central', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 496668, 'npp_pct' => 53.22, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 405262, 'ndc_pct' => 43.43, 'others_votes' => 45239, 'others_pct' => 3.35, 'total' => 947169, 'winner' => 'NPP'],
            ['year' => 2016, 'type' => 'presidential', 'region' => 'Eastern', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 678482, 'npp_pct' => 62.38, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 397549, 'ndc_pct' => 36.55, 'others_votes' => 24870, 'others_pct' => 1.07, 'total' => 1100901, 'winner' => 'NPP'],
            ['year' => 2016, 'type' => 'presidential', 'region' => 'Greater Accra', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 1062157, 'npp_pct' => 52.42, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 946048, 'ndc_pct' => 46.69, 'others_votes' => 18980, 'others_pct' => 0.89, 'total' => 2027185, 'winner' => 'NPP'],
            ['year' => 2016, 'type' => 'presidential', 'region' => 'North East', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 94068, 'npp_pct' => 49.38, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 92395, 'ndc_pct' => 48.50, 'others_votes' => 9411, 'others_pct' => 2.12, 'total' => 195874, 'winner' => 'NPP'],
            ['year' => 2016, 'type' => 'presidential', 'region' => 'Northern', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 286868, 'npp_pct' => 41.34, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 389132, 'ndc_pct' => 56.08, 'others_votes' => 31818, 'others_pct' => 2.58, 'total' => 707818, 'winner' => 'NDC'],
            ['year' => 2016, 'type' => 'presidential', 'region' => 'Oti', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 74275, 'npp_pct' => 34.58, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 136257, 'ndc_pct' => 63.43, 'others_votes' => 10258, 'others_pct' => 1.99, 'total' => 220790, 'winner' => 'NDC'],
            ['year' => 2016, 'type' => 'presidential', 'region' => 'Savannah', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 55485, 'npp_pct' => 33.40, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 106256, 'ndc_pct' => 63.97, 'others_votes' => 9080, 'others_pct' => 2.63, 'total' => 170821, 'winner' => 'NDC'],
            ['year' => 2016, 'type' => 'presidential', 'region' => 'Upper East', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 157398, 'npp_pct' => 34.93, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 271796, 'ndc_pct' => 60.32, 'others_votes' => 37047, 'others_pct' => 4.75, 'total' => 466241, 'winner' => 'NDC'],
            ['year' => 2016, 'type' => 'presidential', 'region' => 'Upper West', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 102843, 'npp_pct' => 35.94, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 167032, 'ndc_pct' => 58.37, 'others_votes' => 24969, 'others_pct' => 5.69, 'total' => 294844, 'winner' => 'NDC'],
            ['year' => 2016, 'type' => 'presidential', 'region' => 'Volta', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 60802, 'npp_pct' => 10.81, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 493141, 'ndc_pct' => 87.67, 'others_votes' => 17085, 'others_pct' => 1.52, 'total' => 571028, 'winner' => 'NDC'],
            ['year' => 2016, 'type' => 'presidential', 'region' => 'Western', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 384211, 'npp_pct' => 55.83, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 285938, 'ndc_pct' => 41.55, 'others_votes' => 27517, 'others_pct' => 2.62, 'total' => 697666, 'winner' => 'NPP'],
            ['year' => 2016, 'type' => 'presidential', 'region' => 'Western North', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 141948, 'npp_pct' => 44.86, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 169900, 'ndc_pct' => 53.70, 'others_votes' => 12224, 'others_pct' => 1.44, 'total' => 324072, 'winner' => 'NDC'],

            // 2020 Presidential Election
            ['year' => 2020, 'type' => 'presidential', 'region' => 'Ahafo', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 145584, 'npp_pct' => 54.04, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 116485, 'ndc_pct' => 44.04, 'others_votes' => 8169, 'others_pct' => 0.92, 'total' => 270238, 'winner' => 'NPP'],
            ['year' => 2020, 'type' => 'presidential', 'region' => 'Ashanti', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 1795774, 'npp_pct' => 72.78, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 653149, 'ndc_pct' => 26.47, 'others_votes' => 53231, 'others_pct' => 0.75, 'total' => 2502154, 'winner' => 'NPP'],
            ['year' => 2020, 'type' => 'presidential', 'region' => 'Bono', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 292604, 'npp_pct' => 58.23, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 203279, 'ndc_pct' => 40.45, 'others_votes' => 20030, 'others_pct' => 1.32, 'total' => 515913, 'winner' => 'NPP'],
            ['year' => 2020, 'type' => 'presidential', 'region' => 'Bono East', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 200469, 'npp_pct' => 42.49, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 264979, 'ndc_pct' => 56.16, 'others_votes' => 19586, 'others_pct' => 1.35, 'total' => 485034, 'winner' => 'NDC'],
            ['year' => 2020, 'type' => 'presidential', 'region' => 'Central', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 613824, 'npp_pct' => 52.31, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 538829, 'ndc_pct' => 45.92, 'others_votes' => 53017, 'others_pct' => 1.77, 'total' => 1205670, 'winner' => 'NPP'],
            ['year' => 2020, 'type' => 'presidential', 'region' => 'Eastern', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 752061, 'npp_pct' => 60.80, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 470999, 'ndc_pct' => 38.08, 'others_votes' => 53844, 'others_pct' => 1.12, 'total' => 1276904, 'winner' => 'NPP'],
            ['year' => 2020, 'type' => 'presidential', 'region' => 'Greater Accra', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 1253179, 'npp_pct' => 48.15, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 1326489, 'ndc_pct' => 50.97, 'others_votes' => 63211, 'others_pct' => 0.88, 'total' => 2642879, 'winner' => 'NDC'],
            ['year' => 2020, 'type' => 'presidential', 'region' => 'North East', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 122742, 'npp_pct' => 51.33, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 112306, 'ndc_pct' => 46.97, 'others_votes' => 15339, 'others_pct' => 1.70, 'total' => 250387, 'winner' => 'NPP'],
            ['year' => 2020, 'type' => 'presidential', 'region' => 'Northern', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 409063, 'npp_pct' => 45.61, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 476650, 'ndc_pct' => 53.14, 'others_votes' => 44818, 'others_pct' => 1.25, 'total' => 930531, 'winner' => 'NDC'],
            ['year' => 2020, 'type' => 'presidential', 'region' => 'Oti', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 103865, 'npp_pct' => 35.88, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 181021, 'ndc_pct' => 62.53, 'others_votes' => 15376, 'others_pct' => 1.59, 'total' => 300262, 'winner' => 'NDC'],
            ['year' => 2020, 'type' => 'presidential', 'region' => 'Savannah', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 80605, 'npp_pct' => 35.19, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 144244, 'ndc_pct' => 62.97, 'others_votes' => 15381, 'others_pct' => 1.84, 'total' => 240230, 'winner' => 'NDC'],
            ['year' => 2020, 'type' => 'presidential', 'region' => 'Upper East', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 170340, 'npp_pct' => 32.84, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 335502, 'ndc_pct' => 64.68, 'others_votes' => 37063, 'others_pct' => 2.48, 'total' => 542905, 'winner' => 'NDC'],
            ['year' => 2020, 'type' => 'presidential', 'region' => 'Upper West', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 121230, 'npp_pct' => 32.80, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 238972, 'ndc_pct' => 64.65, 'others_votes' => 26615, 'others_pct' => 2.55, 'total' => 386817, 'winner' => 'NDC'],
            ['year' => 2020, 'type' => 'presidential', 'region' => 'Volta', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 101021, 'npp_pct' => 14.12, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 606508, 'ndc_pct' => 84.77, 'others_votes' => 23182, 'others_pct' => 1.11, 'total' => 730711, 'winner' => 'NDC'],
            ['year' => 2020, 'type' => 'presidential', 'region' => 'Western', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 439724, 'npp_pct' => 50.93, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 398549, 'ndc_pct' => 46.16, 'others_votes' => 74349, 'others_pct' => 2.91, 'total' => 912622, 'winner' => 'NPP'],
            ['year' => 2020, 'type' => 'presidential', 'region' => 'Western North', 'npp_candidate' => 'Nana Akufo-Addo', 'npp_votes' => 175240, 'npp_pct' => 46.41, 'ndc_candidate' => 'John Dramani Mahama', 'ndc_votes' => 196556, 'ndc_pct' => 52.05, 'others_votes' => 16954, 'others_pct' => 1.54, 'total' => 388750, 'winner' => 'NDC'],
        ];

        // Get party and candidate IDs
        $nppId = DB::table('parties')->where('short_name', 'NPP')->value('id');
        $ndcId = DB::table('parties')->where('short_name', 'NDC')->value('id');
        $othersId = DB::table('parties')->where('short_name', 'OTH')->value('id');

        foreach ($results as $result) {
            // Get election ID
            $electionId = DB::table('elections')
                ->where('year', $result['year'])
                ->where('type', $result['type'])
                ->value('id');
            
            if (!$electionId) continue;

            // Get region ID
            $regionId = DB::table('regions')
                ->where('name', $result['region'])
                ->value('id');
            
            if (!$regionId) continue;

            // Get or create NPP candidate ID
            $nppCandidateId = DB::table('candidates')
                ->where('name', $result['npp_candidate'])
                ->value('id');

            // Get or create NDC candidate ID
            $ndcCandidateId = DB::table('candidates')
                ->where('name', $result['ndc_candidate'])
                ->value('id');

            // Insert NPP result
            DB::table('election_results_regional')->updateOrInsert(
                [
                    'election_id' => $electionId,
                    'region_id' => $regionId,
                    'party_id' => $nppId,
                ],
                [
                    'candidate_id' => $nppCandidateId,
                    'votes' => $result['npp_votes'],
                    'percentage' => $result['npp_pct'],
                    'total_votes_in_region' => $result['total'],
                    'is_region_winner' => $result['winner'] === 'NPP',
                    'created_at' => now(),
                    'updated_at' => now(),
                ]
            );

            // Insert NDC result
            DB::table('election_results_regional')->updateOrInsert(
                [
                    'election_id' => $electionId,
                    'region_id' => $regionId,
                    'party_id' => $ndcId,
                ],
                [
                    'candidate_id' => $ndcCandidateId,
                    'votes' => $result['ndc_votes'],
                    'percentage' => $result['ndc_pct'],
                    'total_votes_in_region' => $result['total'],
                    'is_region_winner' => $result['winner'] === 'NDC',
                    'created_at' => now(),
                    'updated_at' => now(),
                ]
            );

            // Insert Others result (if any)
            if ($result['others_votes'] > 0) {
                DB::table('election_results_regional')->updateOrInsert(
                    [
                        'election_id' => $electionId,
                        'region_id' => $regionId,
                        'party_id' => $othersId,
                    ],
                    [
                        'candidate_id' => null,
                        'votes' => $result['others_votes'],
                        'percentage' => $result['others_pct'],
                        'total_votes_in_region' => $result['total'],
                        'is_region_winner' => false,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]
                );
            }
        }
    }

    private function seedParliamentarySeats(): void
    {
        // Parliamentary seat data from IPU/Electoral Commission
        $parliamentaryData = [
            // 2000
            ['year' => 2000, 'npp_seats' => 100, 'ndc_seats' => 92, 'others_seats' => 8],
            // 2004
            ['year' => 2004, 'npp_seats' => 128, 'ndc_seats' => 94, 'others_seats' => 8],
            // 2008
            ['year' => 2008, 'npp_seats' => 107, 'ndc_seats' => 116, 'others_seats' => 7],
            // 2012
            ['year' => 2012, 'npp_seats' => 123, 'ndc_seats' => 148, 'others_seats' => 4],
            // 2016
            ['year' => 2016, 'npp_seats' => 169, 'ndc_seats' => 106, 'others_seats' => 0],
            // 2020
            ['year' => 2020, 'npp_seats' => 137, 'ndc_seats' => 137, 'others_seats' => 1],
            // 2024 (placeholder - update with actual results)
            ['year' => 2024, 'npp_seats' => 0, 'ndc_seats' => 0, 'others_seats' => 0],
        ];

        $nppId = DB::table('parties')->where('short_name', 'NPP')->value('id');
        $ndcId = DB::table('parties')->where('short_name', 'NDC')->value('id');
        $othersId = DB::table('parties')->where('short_name', 'OTH')->value('id');

        foreach ($parliamentaryData as $data) {
            $electionId = DB::table('elections')
                ->where('year', $data['year'])
                ->where('type', 'parliamentary')
                ->value('id');
            
            if (!$electionId) continue;

            $totalSeats = $data['npp_seats'] + $data['ndc_seats'] + $data['others_seats'];

            // NPP seats
            if ($data['npp_seats'] > 0) {
                DB::table('parliamentary_seats')->updateOrInsert(
                    ['election_id' => $electionId, 'party_id' => $nppId],
                    [
                        'seats_won' => $data['npp_seats'],
                        'percentage' => $totalSeats > 0 ? round(($data['npp_seats'] / $totalSeats) * 100, 2) : 0,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]
                );
            }

            // NDC seats
            if ($data['ndc_seats'] > 0) {
                DB::table('parliamentary_seats')->updateOrInsert(
                    ['election_id' => $electionId, 'party_id' => $ndcId],
                    [
                        'seats_won' => $data['ndc_seats'],
                        'percentage' => $totalSeats > 0 ? round(($data['ndc_seats'] / $totalSeats) * 100, 2) : 0,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]
                );
            }

            // Others seats
            if ($data['others_seats'] > 0) {
                DB::table('parliamentary_seats')->updateOrInsert(
                    ['election_id' => $electionId, 'party_id' => $othersId],
                    [
                        'seats_won' => $data['others_seats'],
                        'percentage' => $totalSeats > 0 ? round(($data['others_seats'] / $totalSeats) * 100, 2) : 0,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]
                );
            }
        }
    }
}
