/**
 * Centralized Theme Management System
 * Ensures consistent theme handling across all pages
 */

class ThemeManager {
    constructor() {
        this.themeKey = 'theme';
        this.defaultTheme = 'light';
        this.currentTheme = null;
        this.html = document.documentElement;
        
        this.init();
    }
    
    init() {
        // Apply theme immediately to prevent flash
        this.applyThemeFromStorage();
        
        // Listen for system theme changes
        this.setupSystemThemeListener();
        
        // Make methods globally available
        window.ThemeManager = this;
    }
    
    /**
     * Get the current theme from localStorage or system preference
     */
    getCurrentTheme() {
        const savedTheme = localStorage.getItem(this.themeKey);
        
        if (savedTheme) {
            return savedTheme;
        }
        
        // Auto-detect system theme if no saved preference
        if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
            return 'dark';
        }
        
        return this.defaultTheme;
    }
    
    /**
     * Apply theme from localStorage
     */
    applyThemeFromStorage() {
        const theme = this.getCurrentTheme();
        this.setTheme(theme);
    }
    
    /**
     * Set the theme and update DOM
     */
    setTheme(theme) {
        this.currentTheme = theme;
        
        if (theme === 'dark') {
            this.html.setAttribute('data-theme', 'dark');
        } else {
            this.html.removeAttribute('data-theme');
        }
        
        // Update theme toggle buttons if they exist
        this.updateThemeToggles();
        
        // Save to localStorage
        localStorage.setItem(this.themeKey, theme);
        
        // Dispatch custom event for other components
        window.dispatchEvent(new CustomEvent('themeChanged', { 
            detail: { theme: theme } 
        }));
    }
    
    /**
     * Toggle between light and dark themes
     */
    toggleTheme() {
        const newTheme = this.currentTheme === 'light' ? 'dark' : 'light';
        this.setTheme(newTheme);
        return newTheme;
    }
    
    /**
     * Update all theme toggle buttons on the page
     */
    updateThemeToggles() {
        const toggles = document.querySelectorAll('[data-theme-toggle]');
        
        toggles.forEach(toggle => {
            const icon = toggle.querySelector('i');
            const text = toggle.querySelector('[data-theme-text]');
            
            if (icon) {
                if (this.currentTheme === 'dark') {
                    icon.className = 'fas fa-moon';
                    toggle.title = 'Switch to Light Mode';
                } else {
                    icon.className = 'fas fa-sun';
                    toggle.title = 'Switch to Dark Mode';
                }
            }
            
            if (text) {
                text.textContent = this.currentTheme === 'dark' ? 'Dark Mode' : 'Light Mode';
            }
        });
    }
    
    /**
     * Setup system theme change listener
     */
    setupSystemThemeListener() {
        if (window.matchMedia) {
            window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', (e) => {
                // Only auto-update if user hasn't manually set a preference
                const userHasSetTheme = localStorage.getItem(this.themeKey);
                if (!userHasSetTheme) {
                    const newTheme = e.matches ? 'dark' : 'light';
                    this.setTheme(newTheme);
                }
            });
        }
    }
    
    /**
     * Get theme for server-side rendering
     */
    getThemeForServer() {
        return this.currentTheme || this.getCurrentTheme();
    }
    
    /**
     * Update user theme preference on server (for logged-in users)
     */
    async updateUserThemePreference(theme) {
        try {
            const response = await fetch('/user/preferences/update', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                    'X-Requested-With': 'XMLHttpRequest'
                },
                body: JSON.stringify({ theme: theme })
            });
            
            if (response.ok) {
                console.log('Theme preference updated successfully');
                return true;
            }
        } catch (error) {
            console.log('Theme preference update failed:', error);
        }
        return false;
    }
}

// Initialize theme manager when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    window.themeManager = new ThemeManager();
    
    // Setup theme toggle event listeners
    document.addEventListener('click', function(e) {
        const toggle = e.target.closest('[data-theme-toggle]');
        if (toggle) {
            e.preventDefault();
            const newTheme = window.themeManager.toggleTheme();
            
            // Update user preference if logged in
            if (window.themeManager.updateUserThemePreference) {
                window.themeManager.updateUserThemePreference(newTheme);
            }
        }
    });
});

// Export for module systems
if (typeof module !== 'undefined' && module.exports) {
    module.exports = ThemeManager;
}
