@extends('admin.layout')

@section('title', 'Reports & Analytics')

@section('content')
<div class="admin-header">
    <h1 class="mb-0">
        <i class="fas fa-chart-bar"></i> Reports & Analytics
    </h1>
    <p class="text-muted mb-0">Comprehensive insights into your Mews AI chatbot performance</p>
</div>

<!-- Date Filter Form -->
<div class="admin-card mb-4">
    <div class="d-flex justify-content-between align-items-center mb-3">
        <h5><i class="fas fa-filter"></i> Filter Reports by Date Range</h5>
        <span class="badge bg-primary">{{ $startDate }} to {{ $endDate }}</span>
    </div>
    
    <form method="GET" action="{{ route('admin.reports') }}" class="row g-3">
        <div class="col-md-4">
            <label for="start_date" class="form-label">Start Date</label>
            <input type="date" class="form-control" id="start_date" name="start_date" value="{{ $startDate }}" required>
        </div>
        <div class="col-md-4">
            <label for="end_date" class="form-label">End Date</label>
            <input type="date" class="form-control" id="end_date" name="end_date" value="{{ $endDate }}" required>
        </div>
        <div class="col-md-4 d-flex align-items-end">
            <button type="submit" class="btn btn-primary me-2">
                <i class="fas fa-search"></i> Apply Filter
            </button>
            <a href="{{ route('admin.reports') }}" class="btn btn-outline-secondary">
                <i class="fas fa-times"></i> Clear
            </a>
        </div>
    </form>
    
    <!-- Quick Date Range Buttons -->
    <div class="mt-3">
        <small class="text-muted">Quick filters:</small>
        <div class="btn-group btn-group-sm ms-2" role="group">
            <a href="{{ route('admin.reports', ['start_date' => now()->subDays(7)->format('Y-m-d'), 'end_date' => now()->format('Y-m-d')]) }}" 
               class="btn btn-outline-primary {{ request('start_date') == now()->subDays(7)->format('Y-m-d') ? 'active' : '' }}">
                Last 7 Days
            </a>
            <a href="{{ route('admin.reports', ['start_date' => now()->subDays(30)->format('Y-m-d'), 'end_date' => now()->format('Y-m-d')]) }}" 
               class="btn btn-outline-primary {{ request('start_date') == now()->subDays(30)->format('Y-m-d') ? 'active' : '' }}">
                Last 30 Days
            </a>
            <a href="{{ route('admin.reports', ['start_date' => now()->subDays(90)->format('Y-m-d'), 'end_date' => now()->format('Y-m-d')]) }}" 
               class="btn btn-outline-primary {{ request('start_date') == now()->subDays(90)->format('Y-m-d') ? 'active' : '' }}">
                Last 90 Days
            </a>
            <a href="{{ route('admin.reports', ['start_date' => now()->startOfMonth()->format('Y-m-d'), 'end_date' => now()->endOfMonth()->format('Y-m-d')]) }}" 
               class="btn btn-outline-primary {{ request('start_date') == now()->startOfMonth()->format('Y-m-d') ? 'active' : '' }}">
                This Month
            </a>
        </div>
    </div>
</div>

<!-- AI Usage Statistics -->
<div class="row mb-4">
    <div class="col-12 mb-3">
        <h6 class="text-muted">
            <i class="fas fa-calendar-alt"></i> 
            AI Usage Statistics {{ $aiUsageStats['date_range'] ?? '' }}
        </h6>
    </div>
    <div class="col-md-2">
        <div class="stats-card">
            <div class="stats-number">{{ number_format($aiUsageStats['total_queries']) }}</div>
            <div class="stats-label">Total Queries</div>
        </div>
    </div>
    <div class="col-md-2">
        <div class="stats-card">
            <div class="stats-number">{{ number_format($aiUsageStats['news_queries']) }}</div>
            <div class="stats-label">News Queries</div>
        </div>
    </div>
    <div class="col-md-2">
        <div class="stats-card">
            <div class="stats-number">{{ number_format($aiUsageStats['general_conversations']) }}</div>
            <div class="stats-label">Conversations</div>
        </div>
    </div>
    <div class="col-md-2">
        <div class="stats-card">
            <div class="stats-number">{{ $aiUsageStats['avg_response_time'] }}</div>
            <div class="stats-label">Avg Response Time</div>
        </div>
    </div>
    <div class="col-md-2">
        <div class="stats-card">
            <div class="stats-number">{{ $aiUsageStats['most_active_hour'] }}</div>
            <div class="stats-label">Peak Hour</div>
        </div>
    </div>
    <div class="col-md-2">
        <div class="stats-card">
            <div class="stats-number">{{ $aiUsageStats['success_rate'] }}</div>
            <div class="stats-label">Success Rate</div>
        </div>
    </div>
</div>

<!-- Top Queries Report -->
<div class="row">
    <div class="col-md-6">
        <div class="admin-card">
            <div class="d-flex justify-content-between align-items-center mb-3">
                <h5><i class="fas fa-search"></i> Top User Queries</h5>
                <span class="badge bg-primary">{{ $topQueries->count() }} queries</span>
            </div>
            
            @if($topQueries->count() > 0)
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>Rank</th>
                                <th>Query</th>
                                <th>Count</th>
                                <th>Last Asked</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($topQueries as $index => $query)
                            <tr>
                                <td>
                                    <span class="badge bg-{{ $index < 3 ? 'warning' : 'secondary' }}">
                                        #{{ $index + 1 }}
                                    </span>
                                </td>
                                <td>
                                    <strong>{{ $query['query'] }}</strong>
                                </td>
                                <td>
                                    <span class="badge bg-info">{{ $query['count'] }}</span>
                                </td>
                                <td>
                                    <small class="text-muted">{{ $query['last_asked'] }}</small>
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            @else
                <div class="text-center py-4">
                    <i class="fas fa-search fa-3x text-muted mb-3"></i>
                    <h6 class="text-muted">No query data available</h6>
                    <p class="text-muted">Query analytics will appear here once users start interacting with the chatbot</p>
                </div>
            @endif
        </div>
    </div>

    <!-- Top Viewed Articles Report -->
    <div class="col-md-6">
        <div class="admin-card">
            <div class="d-flex justify-content-between align-items-center mb-3">
                <h5><i class="fas fa-eye"></i> Top Viewed Articles</h5>
                <span class="badge bg-success">{{ $topViewedArticles->count() }} articles</span>
            </div>
            
            @if($topViewedArticles->count() > 0)
                <div class="table-responsive" style="max-height: 400px; overflow-y: auto;">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>Rank</th>
                                <th>Article</th>
                                <th>Category</th>
                                <th>Views</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($topViewedArticles->take(15) as $index => $article)
                            <tr>
                                <td>
                                    <span class="badge bg-{{ $index < 3 ? 'warning' : 'secondary' }}">
                                        #{{ $index + 1 }}
                                    </span>
                                </td>
                                <td>
                                    <div>
                                        <strong>{{ Str::limit($article['post_title'], 50) }}</strong>
                                        <br>
                                        <small class="text-muted">{{ $article['source'] }}</small>
                                    </div>
                                </td>
                                <td>
                                    <span class="badge bg-info">{{ $article['category'] }}</span>
                                    @if($article['sub_category'])
                                        <br><small class="text-muted">{{ $article['sub_category'] }}</small>
                                    @endif
                                </td>
                                <td>
                                    <span class="badge bg-success">{{ number_format($article['views']) }}</span>
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            @else
                <div class="text-center py-4">
                    <i class="fas fa-eye fa-3x text-muted mb-3"></i>
                    <h6 class="text-muted">No view data available</h6>
                    <p class="text-muted">Article view statistics will appear here once users start reading articles</p>
                </div>
            @endif
        </div>
    </div>
</div>

<!-- Category Statistics -->
<div class="row">
    <div class="col-md-6">
        <div class="admin-card">
            <div class="d-flex justify-content-between align-items-center mb-3">
                <h5><i class="fas fa-tags"></i> Category Performance</h5>
                <span class="badge bg-primary">{{ $categoryStats->count() }} categories</span>
            </div>
            
            @if($categoryStats->count() > 0)
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>Category</th>
                                <th>Articles</th>
                                <th>Total Views</th>
                                <th>Avg Views</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($categoryStats as $category)
                            <tr>
                                <td>
                                    <strong>{{ $category['category'] }}</strong>
                                </td>
                                <td>
                                    <span class="badge bg-info">{{ number_format($category['article_count']) }}</span>
                                </td>
                                <td>
                                    <span class="badge bg-success">{{ number_format($category['total_views']) }}</span>
                                </td>
                                <td>
                                    <span class="badge bg-warning">{{ number_format($category['avg_views'], 1) }}</span>
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            @else
                <div class="text-center py-4">
                    <i class="fas fa-tags fa-3x text-muted mb-3"></i>
                    <h6 class="text-muted">No category data available</h6>
                </div>
            @endif
        </div>
    </div>

    <!-- Daily Statistics -->
    <div class="col-md-6">
        <div class="admin-card">
            <div class="d-flex justify-content-between align-items-center mb-3">
                <h5><i class="fas fa-calendar-day"></i> Daily Activity (Last 30 Days)</h5>
                <span class="badge bg-primary">{{ $dailyStats->count() }} days</span>
            </div>
            
            @if($dailyStats->count() > 0)
                <div class="table-responsive" style="max-height: 400px; overflow-y: auto;">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Articles</th>
                                <th>Views</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($dailyStats->take(15) as $day)
                            <tr>
                                <td>
                                    <strong>{{ \Carbon\Carbon::parse($day['date'])->format('M j, Y') }}</strong>
                                </td>
                                <td>
                                    <span class="badge bg-info">{{ $day['articles_published'] }}</span>
                                </td>
                                <td>
                                    <span class="badge bg-success">{{ number_format($day['total_views']) }}</span>
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            @else
                <div class="text-center py-4">
                    <i class="fas fa-calendar-day fa-3x text-muted mb-3"></i>
                    <h6 class="text-muted">No daily data available</h6>
                </div>
            @endif
        </div>
    </div>
</div>

<!-- Additional Reports Suggestions -->
<div class="admin-card">
    <h5><i class="fas fa-lightbulb"></i> Additional Reports You Can Implement</h5>
    <div class="row">
        <div class="col-md-4">
            <div class="card border-primary">
                <div class="card-body">
                    <h6 class="card-title text-primary">
                        <i class="fas fa-clock"></i> Time-based Analytics
                    </h6>
                    <p class="card-text small">
                        • Hourly usage patterns<br>
                        • Peak activity times<br>
                        • Response time trends<br>
                        • Session duration analysis
                    </p>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card border-success">
                <div class="card-body">
                    <h6 class="card-title text-success">
                        <i class="fas fa-users"></i> User Behavior
                    </h6>
                    <p class="card-text small">
                        • User engagement metrics<br>
                        • Return visitor rates<br>
                        • Query complexity analysis<br>
                        • Geographic distribution
                    </p>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card border-warning">
                <div class="card-body">
                    <h6 class="card-title text-warning">
                        <i class="fas fa-robot"></i> AI Performance
                    </h6>
                    <p class="card-text small">
                        • Intent classification accuracy<br>
                        • Response quality scores<br>
                        • Error rate analysis<br>
                        • Model performance metrics
                    </p>
                </div>
            </div>
        </div>
    </div>
</div>

@endsection

@section('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Set max date to today
    const today = new Date().toISOString().split('T')[0];
    document.getElementById('end_date').setAttribute('max', today);
    document.getElementById('start_date').setAttribute('max', today);
    
    // Validate date range
    const startDateInput = document.getElementById('start_date');
    const endDateInput = document.getElementById('end_date');
    
    function validateDateRange() {
        const startDate = new Date(startDateInput.value);
        const endDate = new Date(endDateInput.value);
        
        if (startDate > endDate) {
            endDateInput.setCustomValidity('End date must be after start date');
        } else {
            endDateInput.setCustomValidity('');
        }
    }
    
    startDateInput.addEventListener('change', validateDateRange);
    endDateInput.addEventListener('change', validateDateRange);
    
    // Auto-submit form when quick filter buttons are clicked
    document.querySelectorAll('.btn-group .btn').forEach(button => {
        button.addEventListener('click', function(e) {
            if (this.classList.contains('active')) {
                e.preventDefault();
                return;
            }
        });
    });
    
    // Add loading state to form submission
    const form = document.querySelector('form');
    form.addEventListener('submit', function() {
        const submitBtn = this.querySelector('button[type="submit"]');
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Loading...';
        submitBtn.disabled = true;
    });
});
</script>
@endsection


