@extends('layouts.app')

@section('title', 'Poll Tracker - Election Intelligence')

@section('styles')
<style>
    .polls-page {
        min-height: 100vh;
        background: var(--bg-secondary);
        padding-bottom: 40px;
    }

    .polls-header {
        background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
        padding: 20px 16px;
        color: white;
    }

    .polls-header-content {
        max-width: 1200px;
        margin: 0 auto;
    }

    .polls-title {
        font-size: 1.5rem;
        font-weight: 700;
        display: flex;
        align-items: center;
        gap: 8px;
    }

    .polls-content {
        max-width: 1200px;
        margin: 0 auto;
        padding: 16px;
    }

    .poll-card {
        background: var(--bg-primary);
        border-radius: 12px;
        padding: 20px;
        margin-bottom: 16px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.05);
    }

    .poll-card-header {
        display: flex;
        justify-content: space-between;
        align-items: flex-start;
        margin-bottom: 16px;
    }

    .poll-pollster {
        font-size: 1.1rem;
        font-weight: 600;
        color: var(--text-primary);
    }

    .poll-org {
        font-size: 0.85rem;
        color: var(--text-secondary);
    }

    .poll-date-badge {
        font-size: 0.75rem;
        padding: 4px 8px;
        background: var(--bg-secondary);
        border-radius: 4px;
        color: var(--text-secondary);
    }

    .poll-results-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
        gap: 12px;
        margin-bottom: 16px;
    }

    .poll-result-item {
        text-align: center;
        padding: 12px;
        background: var(--bg-secondary);
        border-radius: 8px;
    }

    .poll-result-party {
        font-weight: 600;
        font-size: 0.9rem;
        margin-bottom: 4px;
    }

    .poll-result-party.npp { color: #0057B7; }
    .poll-result-party.ndc { color: #009739; }

    .poll-result-pct {
        font-size: 1.5rem;
        font-weight: 700;
    }

    .poll-meta-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(100px, 1fr));
        gap: 8px;
        padding-top: 12px;
        border-top: 1px solid var(--border-color);
        font-size: 0.8rem;
        color: var(--text-secondary);
    }

    .poll-meta-item strong {
        display: block;
        color: var(--text-primary);
    }

    .poll-disclaimer {
        font-size: 0.75rem;
        color: var(--text-secondary);
        margin-top: 12px;
        padding: 8px;
        background: rgba(255, 193, 7, 0.1);
        border-radius: 4px;
    }

    /* Add Poll Form */
    .add-poll-card {
        background: var(--bg-primary);
        border-radius: 12px;
        padding: 24px;
        margin-bottom: 24px;
    }

    .form-section-title {
        font-size: 1rem;
        font-weight: 600;
        margin-bottom: 16px;
        padding-bottom: 8px;
        border-bottom: 1px solid var(--border-color);
    }

    .poll-result-input {
        display: flex;
        align-items: center;
        gap: 12px;
        margin-bottom: 12px;
    }

    .poll-result-input .party-label {
        width: 60px;
        font-weight: 600;
    }

    .poll-result-input .party-label.npp { color: #0057B7; }
    .poll-result-input .party-label.ndc { color: #009739; }

    .verified-badge {
        font-size: 0.7rem;
        padding: 2px 6px;
        background: #22c55e;
        color: white;
        border-radius: 4px;
    }

    .unverified-badge {
        font-size: 0.7rem;
        padding: 2px 6px;
        background: #f59e0b;
        color: white;
        border-radius: 4px;
    }
</style>
@endsection

@section('content')
<div class="polls-page">
    <div class="polls-header">
        <div class="polls-header-content">
            <a href="{{ route('elections.dashboard') }}" class="text-white text-decoration-none mb-2 d-inline-block" style="font-size: 0.85rem;">
                ← Back to Dashboard
            </a>
            <div class="polls-title">
                <span>📊</span>
                Poll Tracker
            </div>
            <p class="mb-0 mt-2" style="font-size: 0.9rem; opacity: 0.8;">
                Track and compare polls from credible sources
            </p>
        </div>
    </div>

    <div class="polls-content">
        <!-- Filter -->
        <div class="d-flex gap-2 mb-4 flex-wrap">
            <select class="form-select" style="max-width: 200px;" onchange="filterPolls(this.value)">
                <option value="">All Elections</option>
                @foreach($elections as $election)
                    <option value="{{ $election->id }}" {{ $electionId == $election->id ? 'selected' : '' }}>
                        {{ $election->name }}
                    </option>
                @endforeach
            </select>
            <button class="btn btn-primary" data-bs-toggle="collapse" data-bs-target="#addPollForm">
                + Add New Poll
            </button>
        </div>

        <!-- Add Poll Form -->
        <div class="collapse {{ request('add') ? 'show' : '' }}" id="addPollForm">
            <div class="add-poll-card">
                <h5 class="mb-4">Add New Poll</h5>
                
                <form action="{{ route('elections.polls.store') }}" method="POST">
                    @csrf
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Election *</label>
                            <select name="election_id" class="form-select" required>
                                <option value="">Select Election</option>
                                @foreach($elections as $election)
                                    <option value="{{ $election->id }}">{{ $election->name }}</option>
                                @endforeach
                            </select>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Pollster Name *</label>
                            <input type="text" name="pollster_name" class="form-control" placeholder="e.g., Global InfoAnalytics" required>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Organization</label>
                            <input type="text" name="pollster_organization" class="form-control" placeholder="e.g., University of Ghana">
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Date Conducted *</label>
                            <input type="date" name="date_conducted" class="form-control" required>
                        </div>
                        
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Sample Size</label>
                            <input type="number" name="sample_size" class="form-control" placeholder="e.g., 2500">
                        </div>
                        
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Margin of Error (%)</label>
                            <input type="number" step="0.1" name="margin_of_error" class="form-control" placeholder="e.g., 3.0">
                        </div>
                        
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Date Published</label>
                            <input type="date" name="date_published" class="form-control">
                        </div>
                        
                        <div class="col-12 mb-3">
                            <label class="form-label">Methodology</label>
                            <textarea name="methodology" class="form-control" rows="2" placeholder="e.g., Telephone survey, random sampling"></textarea>
                        </div>
                        
                        <div class="col-12 mb-3">
                            <label class="form-label">Source URL</label>
                            <input type="url" name="source_url" class="form-control" placeholder="https://...">
                        </div>
                    </div>
                    
                    <div class="form-section-title">Poll Results *</div>
                    
                    <div class="poll-result-input">
                        <span class="party-label npp">NPP</span>
                        <input type="hidden" name="results[0][party_id]" value="{{ \App\Models\Election\Party::where('short_name', 'NPP')->value('id') }}">
                        <input type="number" step="0.1" name="results[0][percentage]" class="form-control" placeholder="%" style="max-width: 100px;" required>
                        <span>%</span>
                    </div>
                    
                    <div class="poll-result-input">
                        <span class="party-label ndc">NDC</span>
                        <input type="hidden" name="results[1][party_id]" value="{{ \App\Models\Election\Party::where('short_name', 'NDC')->value('id') }}">
                        <input type="number" step="0.1" name="results[1][percentage]" class="form-control" placeholder="%" style="max-width: 100px;" required>
                        <span>%</span>
                    </div>
                    
                    <div class="col-12 mb-3 mt-3">
                        <label class="form-label">Disclaimer</label>
                        <textarea name="disclaimer" class="form-control" rows="2">Poll results are estimates and subject to margin of error. Actual election outcomes may differ.</textarea>
                    </div>
                    
                    <div class="mt-4">
                        <button type="submit" class="btn btn-primary">Submit Poll</button>
                        <button type="button" class="btn btn-outline-secondary" data-bs-toggle="collapse" data-bs-target="#addPollForm">Cancel</button>
                    </div>
                </form>
            </div>
        </div>

        @if(session('success'))
        <div class="alert alert-success alert-dismissible fade show">
            {{ session('success') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        @endif

        <!-- Polls List -->
        @forelse($polls as $poll)
        <div class="poll-card">
            <div class="poll-card-header">
                <div>
                    <div class="poll-pollster">
                        {{ $poll->pollster_name }}
                        @if($poll->is_verified)
                            <span class="verified-badge">✓ Verified</span>
                        @else
                            <span class="unverified-badge">Unverified</span>
                        @endif
                    </div>
                    @if($poll->pollster_organization)
                        <div class="poll-org">{{ $poll->pollster_organization }}</div>
                    @endif
                </div>
                <div class="poll-date-badge">
                    {{ $poll->date_conducted->format('M j, Y') }}
                </div>
            </div>
            
            <div class="poll-results-grid">
                @foreach($poll->results->sortByDesc('percentage') as $result)
                <div class="poll-result-item">
                    <div class="poll-result-party {{ strtolower($result->party->short_name ?? '') }}">
                        {{ $result->party->short_name ?? 'Other' }}
                    </div>
                    <div class="poll-result-pct">{{ number_format($result->percentage, 1) }}%</div>
                </div>
                @endforeach
            </div>
            
            <div class="poll-meta-grid">
                @if($poll->sample_size)
                <div class="poll-meta-item">
                    <strong>Sample Size</strong>
                    {{ number_format($poll->sample_size) }}
                </div>
                @endif
                
                @if($poll->margin_of_error)
                <div class="poll-meta-item">
                    <strong>Margin of Error</strong>
                    ±{{ $poll->margin_of_error }}%
                </div>
                @endif
                
                @if($poll->methodology)
                <div class="poll-meta-item">
                    <strong>Methodology</strong>
                    {{ Str::limit($poll->methodology, 50) }}
                </div>
                @endif
                
                @if($poll->source_url)
                <div class="poll-meta-item">
                    <strong>Source</strong>
                    <a href="{{ $poll->source_url }}" target="_blank" rel="noopener">View →</a>
                </div>
                @endif
            </div>
            
            @if($poll->disclaimer)
            <div class="poll-disclaimer">
                ⚠️ {{ $poll->disclaimer }}
            </div>
            @endif
        </div>
        @empty
        <div class="text-center py-5">
            <div style="font-size: 3rem; margin-bottom: 16px;">📊</div>
            <h5>No Polls Yet</h5>
            <p class="text-muted">Be the first to add a poll from a credible source</p>
            <button class="btn btn-primary" data-bs-toggle="collapse" data-bs-target="#addPollForm">
                + Add Poll
            </button>
        </div>
        @endforelse

        @if($polls->hasPages())
        <div class="d-flex justify-content-center mt-4">
            {{ $polls->links() }}
        </div>
        @endif
    </div>
</div>

<script>
function filterPolls(electionId) {
    const url = new URL(window.location);
    if (electionId) {
        url.searchParams.set('election_id', electionId);
    } else {
        url.searchParams.delete('election_id');
    }
    window.location = url;
}
</script>
@endsection
