<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>OneSignal Test Page</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
            background-color: #f5f5f5;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .status {
            padding: 15px;
            margin: 10px 0;
            border-radius: 5px;
        }
        .success { background-color: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .error { background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .info { background-color: #d1ecf1; color: #0c5460; border: 1px solid #bee5eb; }
        .warning { background-color: #fff3cd; color: #856404; border: 1px solid #ffeaa7; }
        button {
            background-color: #007bff;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 5px;
            cursor: pointer;
            margin: 5px;
        }
        button:hover { background-color: #0056b3; }
        button:disabled { background-color: #6c757d; cursor: not-allowed; }
        .code {
            background-color: #f8f9fa;
            padding: 10px;
            border-radius: 5px;
            font-family: monospace;
            margin: 10px 0;
            border: 1px solid #e9ecef;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔔 OneSignal Test Page</h1>
        <p>This page helps you test OneSignal push notifications and get your player ID.</p>

        <div id="status-container">
            <div class="status info">
                <strong>Step 1:</strong> OneSignal is initializing...
            </div>
        </div>

        <div id="player-info" style="display: none;">
            <h3>Your OneSignal Player ID:</h3>
            <div class="code" id="player-id">Loading...</div>
            <button onclick="copyPlayerId()">Copy Player ID</button>
        </div>

        <div id="notification-status" style="display: none;">
            <h3>Notification Permission:</h3>
            <div class="status" id="permission-status">Checking...</div>
            <button id="request-permission-btn" onclick="requestPermission()" style="display: none;">
                Request Notification Permission
            </button>
        </div>

        <div id="test-section" style="display: none;">
            <h3>Test Notifications:</h3>
            <button onclick="sendTestNotification()">Send Test Notification</button>
            <button onclick="subscribeToNotifications()">Subscribe to Notifications</button>
        </div>

        <div id="debug-info">
            <h3>Debug Information:</h3>
            <div class="code" id="debug-content">Loading debug info...</div>
        </div>
    </div>

    <!-- OneSignal SDK -->
    <script src="https://cdn.onesignal.com/sdks/web/v16/OneSignalSDK.page.js"></script>
    
    <script>
        let playerId = null;
        let permissionStatus = 'unknown';
        let initializationAttempts = 0;
        const maxAttempts = 10;

        // Add error handling
        window.addEventListener('error', function(e) {
            console.error('JavaScript Error:', e.error);
            updateStatus('error', '❌ JavaScript Error: ' + e.message);
        });

        // Initialize OneSignal with retry logic
        function initializeOneSignal() {
            initializationAttempts++;
            console.log('OneSignal initialization attempt:', initializationAttempts);
            
            if (initializationAttempts > maxAttempts) {
                updateStatus('error', '❌ OneSignal failed to initialize after ' + maxAttempts + ' attempts. Check console for errors.');
                return;
            }

            try {
                window.OneSignal = window.OneSignal || [];
                OneSignal.push(function() {
                    console.log('OneSignal SDK loaded, initializing...');
                    
                    OneSignal.init({
                        appId: "{{ config('services.onesignal.app_id') }}",
                        safari_web_id: "{{ config('services.onesignal.safari_web_id') }}",
                        notifyButton: {
                            enable: false,
                        },
                        allowLocalhostAsSecureOrigin: true,
                        subdomainName: "mewsai",
                    }).then(function() {
                        console.log('OneSignal init completed');
                        updateStatus('success', '✅ OneSignal initialized successfully!');
                        
                        // Get player ID
                        return OneSignal.getUserId();
                    }).then(function(userId) {
                        console.log('Player ID:', userId);
                        if (userId) {
                            playerId = userId;
                            document.getElementById('player-id').textContent = userId;
                            document.getElementById('player-info').style.display = 'block';
                            updateStatus('success', '✅ OneSignal initialized successfully! Player ID: ' + userId);
                            
                            // Store in localStorage
                            localStorage.setItem('onesignal_player_id', userId);
                            
                            // Send player ID to server
                            sendPlayerIdToServer(userId);
                        } else {
                            updateStatus('warning', '⚠️ OneSignal initialized but no player ID yet. Request permission below.');
                        }
                        
                        document.getElementById('test-section').style.display = 'block';
                        
                        // Check notification permission
                        return OneSignal.getNotificationPermission();
                    }).then(function(permission) {
                        console.log('Permission status:', permission);
                        permissionStatus = permission;
                        updatePermissionStatus(permission);
                        document.getElementById('notification-status').style.display = 'block';
                    }).catch(function(error) {
                        console.error('OneSignal initialization error:', error);
                        updateStatus('error', '❌ OneSignal initialization failed: ' + error.message);
                        
                        // Retry after 2 seconds
                        setTimeout(initializeOneSignal, 2000);
                    });
                });
            } catch (error) {
                console.error('OneSignal setup error:', error);
                updateStatus('error', '❌ OneSignal setup failed: ' + error.message);
                
                // Retry after 2 seconds
                setTimeout(initializeOneSignal, 2000);
            }
        }

        // Start initialization when page loads
        document.addEventListener('DOMContentLoaded', function() {
            console.log('DOM loaded, starting OneSignal initialization...');
            updateStatus('info', '🔄 Initializing OneSignal... (Attempt ' + initializationAttempts + ')');
            
            // Wait a bit for the SDK to load
            setTimeout(initializeOneSignal, 1000);
        });

        function updateStatus(type, message) {
            const container = document.getElementById('status-container');
            container.innerHTML = `<div class="status ${type}">${message}</div>`;
        }

        function updatePermissionStatus(permission) {
            const statusDiv = document.getElementById('permission-status');
            const button = document.getElementById('request-permission-btn');
            
            switch(permission) {
                case 'granted':
                    statusDiv.className = 'status success';
                    statusDiv.innerHTML = '✅ Notifications are enabled!';
                    button.style.display = 'none';
                    break;
                case 'denied':
                    statusDiv.className = 'status error';
                    statusDiv.innerHTML = '❌ Notifications are blocked. Please enable them in your browser settings.';
                    button.style.display = 'none';
                    break;
                default:
                    statusDiv.className = 'status warning';
                    statusDiv.innerHTML = '⚠️ Notification permission not determined. Click the button below to request permission.';
                    button.style.display = 'block';
            }
        }

        function requestPermission() {
            OneSignal.showNativePrompt().then(function() {
                // Permission granted, refresh the page to get player ID
                location.reload();
            });
        }

        function copyPlayerId() {
            navigator.clipboard.writeText(playerId).then(function() {
                alert('Player ID copied to clipboard!');
            });
        }

        function sendPlayerIdToServer(playerId) {
            fetch('/api/onesignal/subscribe', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
                },
                body: JSON.stringify({
                    player_id: playerId,
                    device_type: 'web',
                    browser: navigator.userAgent,
                    timezone: Intl.DateTimeFormat().resolvedOptions().timeZone
                })
            }).then(response => response.json())
            .then(data => {
                console.log('Player ID sent to server:', data);
            }).catch(error => {
                console.error('Error sending player ID to server:', error);
            });
        }

        function sendTestNotification() {
            if (!playerId) {
                alert('No player ID available. Please allow notifications first.');
                return;
            }

            // Send a test notification to yourself
            fetch('/api/test-notification', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
                },
                body: JSON.stringify({
                    player_id: playerId,
                    title: 'Test Notification',
                    message: 'This is a test notification from Mews!'
                })
            }).then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Test notification sent! Check your notifications.');
                } else {
                    alert('Failed to send test notification: ' + data.message);
                }
            }).catch(error => {
                alert('Error sending test notification: ' + error.message);
            });
        }

        function subscribeToNotifications() {
            if (!playerId) {
                alert('No player ID available. Please allow notifications first.');
                return;
            }

            // Update notification preferences
            fetch('/api/onesignal/preferences', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
                },
                body: JSON.stringify({
                    player_id: playerId,
                    breaking_news_enabled: true,
                    daily_digest_enabled: true,
                    weekly_summary_enabled: true,
                    marketing_enabled: false,
                    subscribed_categories: ['politics', 'business', 'technology', 'sports'],
                    subscribed_locations: ['ghana', 'accra']
                })
            }).then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Successfully subscribed to notifications!');
                } else {
                    alert('Failed to subscribe: ' + data.message);
                }
            }).catch(error => {
                alert('Error subscribing: ' + error.message);
            });
        }

        // Update debug info
        function updateDebugInfo() {
            const debugContent = document.getElementById('debug-content');
            debugContent.innerHTML = `
App ID: {{ config('services.onesignal.app_id') }}
Player ID: ${playerId || 'Not available'}
Permission: ${permissionStatus}
User Agent: ${navigator.userAgent}
Timezone: ${Intl.DateTimeFormat().resolvedOptions().timeZone}
Local Storage Player ID: ${localStorage.getItem('onesignal_player_id') || 'Not set'}
            `;
        }

        // Update debug info every second
        setInterval(updateDebugInfo, 1000);
    </script>
</body>
</html>

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>OneSignal Test Page</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
            background-color: #f5f5f5;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .status {
            padding: 15px;
            margin: 10px 0;
            border-radius: 5px;
        }
        .success { background-color: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .error { background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .info { background-color: #d1ecf1; color: #0c5460; border: 1px solid #bee5eb; }
        .warning { background-color: #fff3cd; color: #856404; border: 1px solid #ffeaa7; }
        button {
            background-color: #007bff;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 5px;
            cursor: pointer;
            margin: 5px;
        }
        button:hover { background-color: #0056b3; }
        button:disabled { background-color: #6c757d; cursor: not-allowed; }
        .code {
            background-color: #f8f9fa;
            padding: 10px;
            border-radius: 5px;
            font-family: monospace;
            margin: 10px 0;
            border: 1px solid #e9ecef;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔔 OneSignal Test Page</h1>
        <p>This page helps you test OneSignal push notifications and get your player ID.</p>

        <div id="status-container">
            <div class="status info">
                <strong>Step 1:</strong> OneSignal is initializing...
            </div>
        </div>

        <div id="player-info" style="display: none;">
            <h3>Your OneSignal Player ID:</h3>
            <div class="code" id="player-id">Loading...</div>
            <button onclick="copyPlayerId()">Copy Player ID</button>
        </div>

        <div id="notification-status" style="display: none;">
            <h3>Notification Permission:</h3>
            <div class="status" id="permission-status">Checking...</div>
            <button id="request-permission-btn" onclick="requestPermission()" style="display: none;">
                Request Notification Permission
            </button>
        </div>

        <div id="test-section" style="display: none;">
            <h3>Test Notifications:</h3>
            <button onclick="sendTestNotification()">Send Test Notification</button>
            <button onclick="subscribeToNotifications()">Subscribe to Notifications</button>
        </div>

        <div id="debug-info">
            <h3>Debug Information:</h3>
            <div class="code" id="debug-content">Loading debug info...</div>
        </div>
    </div>

    <!-- OneSignal SDK -->
    <script src="https://cdn.onesignal.com/sdks/web/v16/OneSignalSDK.page.js"></script>
    
    <script>
        let playerId = null;
        let permissionStatus = 'unknown';
        let initializationAttempts = 0;
        const maxAttempts = 10;

        // Add error handling
        window.addEventListener('error', function(e) {
            console.error('JavaScript Error:', e.error);
            updateStatus('error', '❌ JavaScript Error: ' + e.message);
        });

        // Initialize OneSignal with retry logic
        function initializeOneSignal() {
            initializationAttempts++;
            console.log('OneSignal initialization attempt:', initializationAttempts);
            
            if (initializationAttempts > maxAttempts) {
                updateStatus('error', '❌ OneSignal failed to initialize after ' + maxAttempts + ' attempts. Check console for errors.');
                return;
            }

            try {
                window.OneSignal = window.OneSignal || [];
                OneSignal.push(function() {
                    console.log('OneSignal SDK loaded, initializing...');
                    
                    OneSignal.init({
                        appId: "{{ config('services.onesignal.app_id') }}",
                        safari_web_id: "{{ config('services.onesignal.safari_web_id') }}",
                        notifyButton: {
                            enable: false,
                        },
                        allowLocalhostAsSecureOrigin: true,
                        subdomainName: "mewsai",
                    }).then(function() {
                        console.log('OneSignal init completed');
                        updateStatus('success', '✅ OneSignal initialized successfully!');
                        
                        // Get player ID
                        return OneSignal.getUserId();
                    }).then(function(userId) {
                        console.log('Player ID:', userId);
                        if (userId) {
                            playerId = userId;
                            document.getElementById('player-id').textContent = userId;
                            document.getElementById('player-info').style.display = 'block';
                            updateStatus('success', '✅ OneSignal initialized successfully! Player ID: ' + userId);
                            
                            // Store in localStorage
                            localStorage.setItem('onesignal_player_id', userId);
                            
                            // Send player ID to server
                            sendPlayerIdToServer(userId);
                        } else {
                            updateStatus('warning', '⚠️ OneSignal initialized but no player ID yet. Request permission below.');
                        }
                        
                        document.getElementById('test-section').style.display = 'block';
                        
                        // Check notification permission
                        return OneSignal.getNotificationPermission();
                    }).then(function(permission) {
                        console.log('Permission status:', permission);
                        permissionStatus = permission;
                        updatePermissionStatus(permission);
                        document.getElementById('notification-status').style.display = 'block';
                    }).catch(function(error) {
                        console.error('OneSignal initialization error:', error);
                        updateStatus('error', '❌ OneSignal initialization failed: ' + error.message);
                        
                        // Retry after 2 seconds
                        setTimeout(initializeOneSignal, 2000);
                    });
                });
            } catch (error) {
                console.error('OneSignal setup error:', error);
                updateStatus('error', '❌ OneSignal setup failed: ' + error.message);
                
                // Retry after 2 seconds
                setTimeout(initializeOneSignal, 2000);
            }
        }

        // Start initialization when page loads
        document.addEventListener('DOMContentLoaded', function() {
            console.log('DOM loaded, starting OneSignal initialization...');
            updateStatus('info', '🔄 Initializing OneSignal... (Attempt ' + initializationAttempts + ')');
            
            // Wait a bit for the SDK to load
            setTimeout(initializeOneSignal, 1000);
        });

        function updateStatus(type, message) {
            const container = document.getElementById('status-container');
            container.innerHTML = `<div class="status ${type}">${message}</div>`;
        }

        function updatePermissionStatus(permission) {
            const statusDiv = document.getElementById('permission-status');
            const button = document.getElementById('request-permission-btn');
            
            switch(permission) {
                case 'granted':
                    statusDiv.className = 'status success';
                    statusDiv.innerHTML = '✅ Notifications are enabled!';
                    button.style.display = 'none';
                    break;
                case 'denied':
                    statusDiv.className = 'status error';
                    statusDiv.innerHTML = '❌ Notifications are blocked. Please enable them in your browser settings.';
                    button.style.display = 'none';
                    break;
                default:
                    statusDiv.className = 'status warning';
                    statusDiv.innerHTML = '⚠️ Notification permission not determined. Click the button below to request permission.';
                    button.style.display = 'block';
            }
        }

        function requestPermission() {
            OneSignal.showNativePrompt().then(function() {
                // Permission granted, refresh the page to get player ID
                location.reload();
            });
        }

        function copyPlayerId() {
            navigator.clipboard.writeText(playerId).then(function() {
                alert('Player ID copied to clipboard!');
            });
        }

        function sendPlayerIdToServer(playerId) {
            fetch('/api/onesignal/subscribe', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
                },
                body: JSON.stringify({
                    player_id: playerId,
                    device_type: 'web',
                    browser: navigator.userAgent,
                    timezone: Intl.DateTimeFormat().resolvedOptions().timeZone
                })
            }).then(response => response.json())
            .then(data => {
                console.log('Player ID sent to server:', data);
            }).catch(error => {
                console.error('Error sending player ID to server:', error);
            });
        }

        function sendTestNotification() {
            if (!playerId) {
                alert('No player ID available. Please allow notifications first.');
                return;
            }

            // Send a test notification to yourself
            fetch('/api/test-notification', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
                },
                body: JSON.stringify({
                    player_id: playerId,
                    title: 'Test Notification',
                    message: 'This is a test notification from Mews!'
                })
            }).then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Test notification sent! Check your notifications.');
                } else {
                    alert('Failed to send test notification: ' + data.message);
                }
            }).catch(error => {
                alert('Error sending test notification: ' + error.message);
            });
        }

        function subscribeToNotifications() {
            if (!playerId) {
                alert('No player ID available. Please allow notifications first.');
                return;
            }

            // Update notification preferences
            fetch('/api/onesignal/preferences', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
                },
                body: JSON.stringify({
                    player_id: playerId,
                    breaking_news_enabled: true,
                    daily_digest_enabled: true,
                    weekly_summary_enabled: true,
                    marketing_enabled: false,
                    subscribed_categories: ['politics', 'business', 'technology', 'sports'],
                    subscribed_locations: ['ghana', 'accra']
                })
            }).then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Successfully subscribed to notifications!');
                } else {
                    alert('Failed to subscribe: ' + data.message);
                }
            }).catch(error => {
                alert('Error subscribing: ' + error.message);
            });
        }

        // Update debug info
        function updateDebugInfo() {
            const debugContent = document.getElementById('debug-content');
            debugContent.innerHTML = `
App ID: {{ config('services.onesignal.app_id') }}
Player ID: ${playerId || 'Not available'}
Permission: ${permissionStatus}
User Agent: ${navigator.userAgent}
Timezone: ${Intl.DateTimeFormat().resolvedOptions().timeZone}
Local Storage Player ID: ${localStorage.getItem('onesignal_player_id') || 'Not set'}
            `;
        }

        // Update debug info every second
        setInterval(updateDebugInfo, 1000);
    </script>
</body>
</html>
