@extends('layouts.user-dashboard')

@section('title', 'Bookmarks')

@section('content')
<div class="mobile-page">
    <!-- Page Header -->
    <div class="page-header">
        <div class="header-content">
            <a href="{{ route('user.dashboard') }}" class="back-button">
                <i class="fas fa-arrow-left"></i>
            </a>
            <div class="header-text">
                <h1 class="page-title">Bookmarks</h1>
                <p class="page-subtitle">Your saved articles</p>
            </div>
        </div>
    </div>

    <!-- Bookmarks Content -->
    <div class="page-content">
        @if($bookmarks->count() > 0)
            <div class="bookmarks-list">
                @foreach($bookmarks as $bookmark)
                    <div class="bookmark-item">
                        <div class="bookmark-icon">
                            <i class="fas fa-bookmark"></i>
                        </div>
                        <div class="bookmark-details">
                            <div class="bookmark-title">{{ $bookmark->article_title }}</div>
                            <div class="bookmark-meta">
                                <span class="bookmark-category">{{ $bookmark->article_category }}</span>
                                <span class="bookmark-source">{{ $bookmark->article_source }}</span>
                                <span class="bookmark-time">{{ $bookmark->bookmarked_at->diffForHumans() }}</span>
                            </div>
                            @if($bookmark->notes)
                                <div class="bookmark-notes">
                                    <i class="fas fa-sticky-note"></i>
                                    {{ Str::limit($bookmark->notes, 100) }}
                                </div>
                            @endif
                            @if($bookmark->tags && count($bookmark->tags) > 0)
                                <div class="bookmark-tags">
                                    @foreach($bookmark->tags as $tag)
                                        <span class="tag">{{ $tag }}</span>
                                    @endforeach
                                </div>
                            @endif
                        </div>
                        <div class="bookmark-actions">
                            <a href="{{ route('news.deep-dive.show', $bookmark->article_id) }}" class="action-btn">
                                <i class="fas fa-external-link-alt"></i>
                            </a>
                            <button class="action-btn remove" data-article-id="{{ $bookmark->article_id }}">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </div>
                @endforeach
            </div>

            <!-- Pagination -->
            @if($bookmarks->hasPages())
                <div class="pagination-wrapper">
                    {{ $bookmarks->links() }}
                </div>
            @endif
        @else
            <div class="empty-state">
                <div class="empty-icon">
                    <i class="fas fa-bookmark"></i>
                </div>
                <h3>No Bookmarks</h3>
                <p>You haven't bookmarked any articles yet. Save articles you find interesting!</p>
                <a href="{{ route('home') }}" class="btn btn-primary">
                    <i class="fas fa-home"></i>
                    Start Reading
                </a>
            </div>
        @endif
    </div>
</div>

<style>
/* Mobile Page Styles */
.mobile-page {
    padding: 1rem;
    max-width: 100%;
    margin: 0 auto;
}

/* Page Header */
.page-header {
    margin-bottom: 2rem;
}

.header-content {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.back-button {
    width: 44px;
    height: 44px;
    border-radius: 12px;
    background: var(--bg-secondary);
    border: 1px solid var(--border-color);
    display: flex;
    align-items: center;
    justify-content: center;
    color: var(--text-primary);
    text-decoration: none;
    transition: all 0.3s ease;
}

.back-button:hover {
    background: var(--accent-gradient);
    color: white;
    text-decoration: none;
}

.header-text {
    flex: 1;
}

.page-title {
    font-size: 1.5rem;
    font-weight: 700;
    color: var(--text-primary);
    margin: 0;
    background: var(--accent-gradient);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.page-subtitle {
    font-size: 0.9rem;
    color: var(--text-secondary);
    margin: 0.25rem 0 0 0;
}

/* Page Content */
.page-content {
    margin-bottom: 2rem;
}

/* Bookmarks List */
.bookmarks-list {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.bookmark-item {
    background: var(--bg-secondary);
    border-radius: 12px;
    border: 1px solid var(--border-color);
    box-shadow: var(--card-shadow);
    padding: 1.25rem;
    display: flex;
    align-items: flex-start;
    gap: 1rem;
    transition: all 0.3s ease;
}

.bookmark-item:hover {
    transform: translateY(-2px);
    box-shadow: var(--card-shadow-hover);
}

.bookmark-icon {
    width: 40px;
    height: 40px;
    border-radius: 8px;
    background: var(--accent-gradient);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 16px;
    flex-shrink: 0;
}

.bookmark-details {
    flex: 1;
    min-width: 0;
}

.bookmark-title {
    font-size: 1rem;
    font-weight: 600;
    color: var(--text-primary);
    line-height: 1.3;
    margin-bottom: 0.5rem;
}

.bookmark-meta {
    display: flex;
    flex-wrap: wrap;
    gap: 0.5rem;
    align-items: center;
    margin-bottom: 0.5rem;
}

.bookmark-category {
    font-size: 0.8rem;
    color: var(--accent-color);
    background: rgba(255, 107, 53, 0.1);
    padding: 0.25rem 0.5rem;
    border-radius: 4px;
    font-weight: 500;
}

.bookmark-source {
    font-size: 0.8rem;
    color: var(--text-secondary);
    font-weight: 500;
}

.bookmark-time {
    font-size: 0.8rem;
    color: var(--text-secondary);
}

.bookmark-notes {
    font-size: 0.9rem;
    color: var(--text-secondary);
    background: rgba(255, 107, 53, 0.05);
    padding: 0.5rem;
    border-radius: 6px;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: flex-start;
    gap: 0.5rem;
}

.bookmark-notes i {
    color: var(--accent-color);
    margin-top: 0.1rem;
    flex-shrink: 0;
}

.bookmark-tags {
    display: flex;
    flex-wrap: wrap;
    gap: 0.5rem;
}

.tag {
    font-size: 0.75rem;
    color: var(--accent-color);
    background: rgba(255, 107, 53, 0.1);
    padding: 0.25rem 0.5rem;
    border-radius: 12px;
    font-weight: 500;
}

.bookmark-actions {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
    flex-shrink: 0;
}

.action-btn {
    width: 36px;
    height: 36px;
    border-radius: 8px;
    background: var(--bg-primary);
    border: 1px solid var(--border-color);
    display: flex;
    align-items: center;
    justify-content: center;
    color: var(--text-secondary);
    text-decoration: none;
    transition: all 0.3s ease;
    cursor: pointer;
}

.action-btn:hover {
    background: var(--accent-gradient);
    color: white;
    text-decoration: none;
    transform: scale(1.1);
}


.action-btn.remove:hover {
    background: #dc3545;
    color: white;
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 3rem 1.5rem;
    color: var(--text-secondary);
}

.empty-icon {
    width: 80px;
    height: 80px;
    border-radius: 50%;
    background: var(--accent-gradient);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 2rem;
    margin: 0 auto 1.5rem;
}

.empty-state h3 {
    font-size: 1.25rem;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 0.5rem;
}

.empty-state p {
    font-size: 1rem;
    margin-bottom: 1.5rem;
    line-height: 1.5;
}

/* Pagination */
.pagination-wrapper {
    margin-top: 2rem;
    display: flex;
    justify-content: center;
}

.pagination {
    display: flex;
    gap: 0.5rem;
    align-items: center;
}

.pagination .page-link {
    background: var(--bg-secondary);
    border: 1px solid var(--border-color);
    color: var(--text-primary);
    padding: 0.5rem 0.75rem;
    border-radius: 8px;
    text-decoration: none;
    transition: all 0.3s ease;
}

.pagination .page-link:hover {
    background: var(--accent-gradient);
    color: white;
    text-decoration: none;
}

.pagination .page-item.active .page-link {
    background: var(--accent-gradient);
    color: white;
    border-color: var(--accent-color);
}

/* Responsive Design */
@media (max-width: 480px) {
    .mobile-page {
        padding: 0.75rem;
    }
    
    .bookmark-item {
        flex-direction: column;
        align-items: flex-start;
        gap: 1rem;
    }
    
    .bookmark-actions {
        flex-direction: row;
        align-self: flex-end;
    }
    
    .bookmark-meta {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.25rem;
    }
}

@media (min-width: 768px) {
    .mobile-page {
        max-width: 600px;
        margin: 0 auto;
    }
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    
    // Handle bookmark removal
    document.querySelectorAll('.action-btn.remove').forEach(button => {
        button.addEventListener('click', function() {
            const articleId = this.dataset.articleId;
            
            if (confirm('Are you sure you want to remove this bookmark?')) {
                fetch('{{ route("user.bookmark.remove") }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    },
                    body: JSON.stringify({
                        article_id: articleId
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        this.closest('.bookmark-item').remove();
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                });
            }
        });
    });
});
</script>
@endsection

