@extends('layouts.user-dashboard')

@section('title', 'Edit Profile')

@section('content')
<div class="mobile-page">
    <!-- Page Header -->
    <div class="page-header">
        <div class="header-content">
            <a href="{{ route('user.dashboard') }}" class="back-button">
                <i class="fas fa-arrow-left"></i>
            </a>
            <div class="header-text">
                <h1 class="page-title">Edit Profile</h1>
                <p class="page-subtitle">Update your personal information</p>
            </div>
        </div>
    </div>

    <!-- Profile Edit Content -->
    <div class="page-content">
        <form method="POST" action="{{ route('user.profile.update') }}" class="profile-form" enctype="multipart/form-data">
            @csrf
            
            <!-- Profile Picture -->
            <div class="form-section">
                <h3 class="section-title">
                    <i class="fas fa-camera"></i>
                    Profile Picture
                </h3>
                
                <div class="avatar-section">
                    <div class="current-avatar">
                        @if($user->profile)
                            <img src="{{ $user->profile->avatar_url }}" 
                                 alt="Current Avatar" 
                                 class="profile-avatar-large">
                        @else
                            <div class="profile-avatar-large placeholder">
                                {{ substr($user->name, 0, 1) }}
                            </div>
                        @endif
                    </div>
                    
                    <div class="avatar-actions">
                        <label for="avatar" class="avatar-upload-btn">
                            <i class="fas fa-camera"></i>
                            Change Photo
                        </label>
                        <input type="file" id="avatar" name="avatar" accept="image/*" style="display: none;">
                        @if($user->profile && $user->profile->avatar)
                            <button type="button" class="remove-avatar-btn" onclick="removeAvatar()">
                                <i class="fas fa-trash"></i>
                                Remove
                            </button>
                        @endif
                    </div>
                </div>
            </div>

            <!-- Personal Information -->
            <div class="form-section">
                <h3 class="section-title">
                    <i class="fas fa-user"></i>
                    Personal Information
                </h3>
                
                <div class="form-group">
                    <label for="name" class="form-label">Display Name</label>
                    <input type="text" 
                           id="name" 
                           name="name" 
                           value="{{ old('name', $user->name) }}" 
                           class="form-input @error('name') is-invalid @enderror"
                           placeholder="Enter your display name">
                    @error('name')
                        <div class="error-message">{{ $message }}</div>
                    @enderror
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="first_name" class="form-label">First Name</label>
                        <input type="text" 
                               id="first_name" 
                               name="first_name" 
                               value="{{ old('first_name', $user->profile->first_name ?? '') }}" 
                               class="form-input @error('first_name') is-invalid @enderror"
                               placeholder="First name">
                        @error('first_name')
                            <div class="error-message">{{ $message }}</div>
                        @enderror
                    </div>
                    
                    <div class="form-group">
                        <label for="last_name" class="form-label">Last Name</label>
                        <input type="text" 
                               id="last_name" 
                               name="last_name" 
                               value="{{ old('last_name', $user->profile->last_name ?? '') }}" 
                               class="form-input @error('last_name') is-invalid @enderror"
                               placeholder="Last name">
                        @error('last_name')
                            <div class="error-message">{{ $message }}</div>
                        @enderror
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="phone" class="form-label">Phone Number</label>
                    <input type="tel" 
                           id="phone" 
                           name="phone" 
                           value="{{ old('phone', $user->profile->phone ?? '') }}" 
                           class="form-input @error('phone') is-invalid @enderror"
                           placeholder="Enter your phone number">
                    @error('phone')
                        <div class="error-message">{{ $message }}</div>
                    @enderror
                </div>
                
                <div class="form-group">
                    <label for="location" class="form-label">Location</label>
                    <input type="text" 
                           id="location" 
                           name="location" 
                           value="{{ old('location', $user->profile->location ?? '') }}" 
                           class="form-input @error('location') is-invalid @enderror"
                           placeholder="Enter your location">
                    @error('location')
                        <div class="error-message">{{ $message }}</div>
                    @enderror
                </div>
                
                <div class="form-group">
                    <label for="bio" class="form-label">Bio</label>
                    <textarea id="bio" 
                              name="bio" 
                              class="form-textarea @error('bio') is-invalid @enderror"
                              placeholder="Tell us about yourself"
                              rows="4">{{ old('bio', $user->profile->bio ?? '') }}</textarea>
                    @error('bio')
                        <div class="error-message">{{ $message }}</div>
                    @enderror
                </div>
            </div>

            <!-- Interests -->
            <div class="form-section">
                <h3 class="section-title">
                    <i class="fas fa-heart"></i>
                    Interests
                </h3>
                
                <div class="interests-grid">
                    @php
                        // Use the actual subcategories from crawler database as news interest options
                        $availableInterests = [
                            'Politics', 'Business', 'Entertainment', 'Sports', 
                            'International', 'General', 'Local', 'Technology'
                        ];
                        $userInterests = $user->profile->interests ?? [];
                    @endphp
                    
                    @foreach($availableInterests as $interest)
                        <label class="interest-option">
                            <input type="checkbox" 
                                   name="interests[]" 
                                   value="{{ $interest }}"
                                   {{ in_array($interest, $userInterests) ? 'checked' : '' }}>
                            <span class="interest-label">{{ $interest }}</span>
                        </label>
                    @endforeach
                </div>
            </div>

            <!-- Language Preference -->
            <div class="form-section">
                <h3 class="section-title">
                    <i class="fas fa-globe"></i>
                    Language & Region
                </h3>
                
                <div class="form-group">
                    <label for="language_preference" class="form-label">Language Preference</label>
                    <select id="language_preference" 
                            name="language_preference" 
                            class="form-select @error('language_preference') is-invalid @enderror">
                        <option value="en" {{ old('language_preference', $user->profile->language_preference ?? 'en') == 'en' ? 'selected' : '' }}>English</option>
                        <option value="twi" {{ old('language_preference', $user->profile->language_preference ?? 'en') == 'twi' ? 'selected' : '' }}>Twi</option>
                        <option value="ga" {{ old('language_preference', $user->profile->language_preference ?? 'en') == 'ga' ? 'selected' : '' }}>Ga</option>
                        <option value="ewe" {{ old('language_preference', $user->profile->language_preference ?? 'en') == 'ewe' ? 'selected' : '' }}>Ewe</option>
                    </select>
                    @error('language_preference')
                        <div class="error-message">{{ $message }}</div>
                    @enderror
                </div>
            </div>

            <!-- Save Button -->
            <div class="form-actions">
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-save"></i>
                    Save Changes
                </button>
                <a href="{{ route('user.dashboard') }}" class="btn btn-secondary">
                    <i class="fas fa-times"></i>
                    Cancel
                </a>
            </div>
        </form>
    </div>
</div>

<style>
/* Profile Edit Styles */
.mobile-page {
    padding: 1rem;
    max-width: 100%;
    margin: 0 auto;
}

/* Page Header */
.page-header {
    margin-bottom: 2rem;
}

.header-content {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.back-button {
    width: 44px;
    height: 44px;
    border-radius: 12px;
    background: var(--bg-secondary);
    border: 1px solid var(--border-color);
    display: flex;
    align-items: center;
    justify-content: center;
    color: var(--text-primary);
    text-decoration: none;
    transition: all 0.3s ease;
}

.back-button:hover {
    background: var(--accent-gradient);
    color: white;
    text-decoration: none;
}

.header-text {
    flex: 1;
}

.page-title {
    font-size: 1.5rem;
    font-weight: 700;
    color: var(--text-primary);
    margin: 0;
    background: var(--accent-gradient);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.page-subtitle {
    font-size: 0.9rem;
    color: var(--text-secondary);
    margin: 0.25rem 0 0 0;
}

/* Form Sections */
.form-section {
    background: var(--bg-secondary);
    border-radius: 16px;
    border: 1px solid var(--border-color);
    padding: 1.5rem;
    margin-bottom: 1.5rem;
}

.section-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0 0 1.5rem 0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

/* Avatar Section */
.avatar-section {
    display: flex;
    align-items: center;
    gap: 1.5rem;
}

.current-avatar {
    flex-shrink: 0;
}

.profile-avatar-large {
    width: 80px;
    height: 80px;
    border-radius: 50%;
    object-fit: cover;
    border: 3px solid var(--accent-color);
}

.profile-avatar-large.placeholder {
    background: var(--accent-gradient);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 2rem;
    font-weight: 600;
}

.avatar-actions {
    display: flex;
    flex-direction: column;
    gap: 0.75rem;
}

.avatar-upload-btn {
    background: var(--accent-gradient);
    color: white;
    border: none;
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    font-weight: 500;
    cursor: pointer;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    transition: all 0.3s ease;
}

.avatar-upload-btn:hover {
    transform: translateY(-2px);
    box-shadow: var(--card-shadow-hover);
}

.remove-avatar-btn {
    background: #dc3545;
    color: white;
    border: none;
    padding: 0.5rem 1rem;
    border-radius: 8px;
    font-size: 0.9rem;
    cursor: pointer;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    transition: all 0.3s ease;
}

.remove-avatar-btn:hover {
    background: #c82333;
}

/* Form Elements */
.form-group {
    margin-bottom: 1.5rem;
}

.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1rem;
}

.form-label {
    display: block;
    font-size: 0.9rem;
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 0.5rem;
}

.form-input,
.form-textarea,
.form-select {
    width: 100%;
    padding: 0.75rem 1rem;
    background: var(--bg-primary);
    border: 1px solid var(--border-color);
    border-radius: 8px;
    color: var(--text-primary);
    font-size: 0.9rem;
    transition: all 0.3s ease;
}

.form-input:focus,
.form-textarea:focus,
.form-select:focus {
    outline: none;
    border-color: var(--accent-color);
    box-shadow: 0 0 0 3px rgba(255, 107, 53, 0.1);
}

.form-textarea {
    resize: vertical;
    min-height: 100px;
}

.error-message {
    color: #dc3545;
    font-size: 0.8rem;
    margin-top: 0.25rem;
}

/* Interests Grid */
.interests-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
    gap: 0.75rem;
}

.interest-option {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem;
    background: var(--bg-primary);
    border: 1px solid var(--border-color);
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.3s ease;
}

.interest-option:hover {
    background: rgba(255, 107, 53, 0.05);
    border-color: var(--accent-color);
}

.interest-option input[type="checkbox"] {
    display: none;
}

.interest-option input[type="checkbox"]:checked + .interest-label {
    color: var(--accent-color);
    font-weight: 600;
}

.interest-option input[type="checkbox"]:checked {
    background: var(--accent-gradient);
}

.interest-label {
    font-size: 0.9rem;
    color: var(--text-primary);
    transition: all 0.3s ease;
}

/* Form Actions */
.form-actions {
    display: flex;
    gap: 1rem;
    justify-content: center;
    margin-top: 2rem;
}

.btn {
    padding: 0.75rem 2rem;
    border-radius: 12px;
    font-weight: 600;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    transition: all 0.3s ease;
    border: none;
    cursor: pointer;
}

.btn-primary {
    background: var(--accent-gradient);
    color: white;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: var(--card-shadow-hover);
}

.btn-secondary {
    background: var(--bg-secondary);
    color: var(--text-primary);
    border: 1px solid var(--border-color);
}

.btn-secondary:hover {
    background: var(--bg-primary);
    text-decoration: none;
}

/* Responsive Design */
@media (max-width: 480px) {
    .mobile-page {
        padding: 0.75rem;
    }
    
    .form-row {
        grid-template-columns: 1fr;
    }
    
    .interests-grid {
        grid-template-columns: 1fr;
    }
    
    .avatar-section {
        flex-direction: column;
        text-align: center;
    }
    
    .form-actions {
        flex-direction: column;
    }
}

@media (min-width: 768px) {
    .mobile-page {
        max-width: 600px;
        margin: 0 auto;
    }
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Avatar preview
    const avatarInput = document.getElementById('avatar');
    const currentAvatar = document.querySelector('.current-avatar img, .current-avatar .placeholder');
    
    avatarInput.addEventListener('change', function(e) {
        const file = e.target.files[0];
        if (file) {
            const reader = new FileReader();
            reader.onload = function(e) {
                if (currentAvatar.tagName === 'IMG') {
                    currentAvatar.src = e.target.result;
                } else {
                    // Replace placeholder with image
                    const img = document.createElement('img');
                    img.src = e.target.result;
                    img.className = 'profile-avatar-large';
                    currentAvatar.parentNode.replaceChild(img, currentAvatar);
                }
            };
            reader.readAsDataURL(file);
        }
    });
});

function removeAvatar() {
    if (confirm('Are you sure you want to remove your profile picture?')) {
        // Add a hidden input to indicate avatar removal
        const form = document.querySelector('.profile-form');
        const hiddenInput = document.createElement('input');
        hiddenInput.type = 'hidden';
        hiddenInput.name = 'remove_avatar';
        hiddenInput.value = '1';
        form.appendChild(hiddenInput);
        
        // Reset avatar display
        const currentAvatar = document.querySelector('.current-avatar img, .current-avatar .placeholder');
        if (currentAvatar.tagName === 'IMG') {
            const placeholder = document.createElement('div');
            placeholder.className = 'profile-avatar-large placeholder';
            placeholder.textContent = '{{ substr($user->name, 0, 1) }}';
            currentAvatar.parentNode.replaceChild(placeholder, currentAvatar);
        }
        
        // Clear file input
        document.getElementById('avatar').value = '';
    }
}
</script>
@endsection


